<?php

namespace Drupal\commerce_paratika;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;

/**
 * Paratika API client service.
 */
class ParatikaApiClient
{

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * The logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Test API URL.
   */
  const TEST_API_URL = 'https://test.paratika.com.tr/paratika/api/v2';

  /**
   * Production API URL.
   */
  const PRODUCTION_API_URL = 'https://vpos.paratika.com.tr/paratika/api/v2';

  /**
   * Constructs a ParatikaApiClient object.
   *
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(ClientInterface $http_client, LoggerChannelFactoryInterface $logger_factory, ConfigFactoryInterface $config_factory)
  {
    $this->httpClient = $http_client;
    $this->logger = $logger_factory->get('commerce_paratika');
    $this->configFactory = $config_factory;
  }

  /**
   * Get the API URL based on test mode.
   *
   * @param bool $test_mode
   *   Whether to use test mode.
   *
   * @return string
   *   The API URL.
   */
  public function getApiUrl($test_mode = TRUE)
  {
    return $test_mode ? self::TEST_API_URL : self::PRODUCTION_API_URL;
  }

  /**
   * Create a payment session token.
   *
   * @param array $params
   *   The session parameters.
   *
   * @return array|null
   *   The API response or NULL on failure.
   */
  public function createSessionToken(array $params)
  {
    $this->logger->debug('createSessionToken called - BEFORE adding ACTION');
    $params['ACTION'] = 'SESSIONTOKEN';
    $this->logger->debug('createSessionToken - AFTER adding ACTION: @action', ['@action' => $params['ACTION'] ?? 'NOT SET']);
    return $this->makeRequest($params);
  }

  /**
   * Query available installments.
   *
   * @param array $params
   *   The query parameters.
   *
   * @return array|null
   *   The API response or NULL on failure.
   */
  public function queryInstallments(array $params)
  {
    $params['ACTION'] = 'QUERYINSTALLMENT';
    return $this->makeRequest($params);
  }

  /**
   * Query transaction status.
   *
   * @param array $params
   *   The query parameters.
   *
   * @return array|null
   *   The API response or NULL on failure.
   */
  public function queryTransaction(array $params)
  {
    $params['ACTION'] = 'QUERYTRANSACTION';
    return $this->makeRequest($params);
  }

  /**
   * Process a refund.
   *
   * @param array $params
   *   The refund parameters.
   *
   * @return array|null
   *   The API response or NULL on failure.
   */
  public function refund(array $params)
  {
    $params['ACTION'] = 'REFUND';
    return $this->makeRequest($params);
  }

  /**
   * Cancel/void a transaction.
   *
   * @param array $params
   *   The cancel parameters.
   *
   * @return array|null
   *   The API response or NULL on failure.
   */
  public function cancel(array $params)
  {
    $params['ACTION'] = 'CANCEL';
    return $this->makeRequest($params);
  }

  /**
   * Make an API request to Paratika.
   *
   * @param array $params
   *   The request parameters.
   *
   * @return array|null
   *   The decoded JSON response or NULL on failure.
   */
  protected function makeRequest(array $params)
  {
    $test_mode = $params['test_mode'] ?? TRUE;
    unset($params['test_mode']);

    $url = $this->getApiUrl($test_mode);

    // Log the request parameters for debugging (ALWAYS LOGGING NOW).
    // if ($test_mode) {
    $this->logger->debug('makeRequest received ' . count($params) . ' parameters');
    $this->logger->debug('ACTION parameter: ' . (isset($params['ACTION']) ? $params['ACTION'] : 'NOT SET'));

    $log_message = "Paratika API request parameters:\n";
    foreach ($params as $key => $value) {
      if ($key === 'MERCHANTPASSWORD') {
        $log_message .= "  $key: " . (empty($value) ? 'EMPTY' : '***HIDDEN***') . "\n";
      } else {
        $log_message .= "  $key: $value\n";
      }
    }
    $this->logger->debug($log_message);
    // }

    try {
      $response = $this->httpClient->request('POST', $url, [
        'form_params' => $params,
        'headers' => [
          'Accept' => 'application/json',
        ],
        'timeout' => 30,
      ]);

      $body = (string) $response->getBody();
      $data = json_decode($body, TRUE);

      if (json_last_error() !== JSON_ERROR_NONE) {
        $this->logger->error('Invalid JSON response from Paratika: @body', ['@body' => $body]);
        return NULL;
      }

      // Log the response for debugging (in test mode only).
      if ($test_mode) {
        $this->logger->debug('Paratika API response: @response', ['@response' => print_r($data, TRUE)]);
      }

      return $data;
    } catch (RequestException $e) {
      $this->logger->error('Paratika API request failed: @message', ['@message' => $e->getMessage()]);
      return NULL;
    }
  }
}
