<?php

namespace Drupal\commerce_paratika\Controller;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Exception\PaymentGatewayException;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for Paratika payment callbacks.
 */
class ParatikaController extends ControllerBase
{

    /**
     * The Paratika API client.
     *
     * @var \Drupal\commerce_paratika\ParatikaApiClient
     */
    protected $apiClient;

    /**
     * The cart session.
     *
     * @var \Drupal\commerce_cart\CartSessionInterface
     */
    protected $cartSession;

    /**
     * {@inheritdoc}
     */
    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container)
    {
        $instance = parent::create($container);
        $instance->apiClient = $container->get('commerce_paratika.api_client');
        return $instance;
    }

    /**
     * Handles the 3DS return callback.
     *
     * @param \Drupal\commerce_order\Entity\OrderInterface $commerce_order
     *   The order.
     * @param \Symfony\Component\HttpFoundation\Request $request
     *   The request.
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     *   A redirect response.
     */
    public function threeDSReturn(OrderInterface $commerce_order, Request $request)
    {
        // If this is a POST request (from Paratika), redirect to GET to restore session.
        if ($request->isMethod('POST')) {
            $pg_tran_id = $request->request->get('pgTranId');
            $is_popup = $request->request->get('is_popup');

            return $this->redirect('commerce_paratika.3ds_return', [
                'commerce_order' => $commerce_order->id(),
            ], [
                'query' => [
                    'pgTranId' => $pg_tran_id,
                    'is_popup' => $is_popup,
                ],
            ]);
        }

        $is_popup = $request->query->get('is_popup');

        // Get the transaction ID from the request.
        $pg_tran_id = $request->query->get('pgTranId');

        if (empty($pg_tran_id)) {
            $this->messenger()->addError($this->t('Payment verification failed. Transaction ID is missing.'));
            $redirect_url = Url::fromRoute('commerce_checkout.form', [
                'commerce_order' => $commerce_order->id(),
                'step' => 'payment',
            ])->toString();

            if ($is_popup) {
                return $this->closePopupAndRedirect($redirect_url);
            }
            return new RedirectResponse($redirect_url);
        }

        // Store the transaction ID in the order data.
        $commerce_order->setData('paratika', ['pgTranId' => $pg_tran_id]);
        $commerce_order->save();

        try {
            // Get the payment gateway.
            $payment_gateway = $commerce_order->get('payment_gateway')->entity;
            if (!$payment_gateway) {
                throw new PaymentGatewayException('Payment gateway not found.');
            }

            /** @var \Drupal\commerce_paratika\Plugin\Commerce\PaymentGateway\Paratika $gateway_plugin */
            $gateway_plugin = $payment_gateway->getPlugin();

            // Create the payment entity.
            $payment_storage = $this->entityTypeManager()->getStorage('commerce_payment');
            $payment = $payment_storage->create([
                'state' => 'new',
                'amount' => $commerce_order->getTotalPrice(),
                'payment_gateway' => $payment_gateway->id(),
                'order_id' => $commerce_order->id(),
            ]);
            $payment->save();

            // Process the payment (this will verify with Paratika).
            $gateway_plugin->createPayment($payment, TRUE);

            // Place the order.
            $transition = $commerce_order->getState()->getWorkflow()->getTransition('place');
            if ($transition) {
                $commerce_order->getState()->applyTransition($transition);
                $commerce_order->save();
            }

            $this->messenger()->addStatus($this->t('Payment was processed successfully.'));

            // Redirect to the next checkout step.
            $redirect_url = Url::fromRoute('commerce_checkout.form', [
                'commerce_order' => $commerce_order->id(),
                'step' => 'complete',
            ])->toString();

            if ($is_popup) {
                return $this->closePopupAndRedirect($redirect_url);
            }
            return new RedirectResponse($redirect_url);
        } catch (\Exception $e) {
            $this->getLogger('commerce_paratika')->error('Payment processing failed: @message', ['@message' => $e->getMessage()]);
            $this->messenger()->addError($this->t('Payment processing failed. Please try again.'));

            $redirect_url = Url::fromRoute('commerce_checkout.form', [
                'commerce_order' => $commerce_order->id(),
                'step' => 'payment',
            ])->toString();

            if ($is_popup) {
                return $this->closePopupAndRedirect($redirect_url);
            }
            return new RedirectResponse($redirect_url);
        }
    }

    /**
     * Handles the 3DS cancel callback.
     *
     * @param \Drupal\commerce_order\Entity\OrderInterface $commerce_order
     *   The order.
     * @param \Symfony\Component\HttpFoundation\Request $request
     *   The request.
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|\Symfony\Component\HttpFoundation\Response
     *   A redirect response or a response to close the popup.
     */
    public function threeDSCancel(OrderInterface $commerce_order, Request $request)
    {
        $is_popup = $request->query->get('is_popup');
        $this->messenger()->addWarning($this->t('Payment was cancelled. You may try again when ready.'));

        $redirect_url = Url::fromRoute('commerce_checkout.form', [
            'commerce_order' => $commerce_order->id(),
            'step' => 'payment',
        ])->toString();

        if ($is_popup) {
            return $this->closePopupAndRedirect($redirect_url);
        }
        return new RedirectResponse($redirect_url);
    }

    /**
     * Helper to close popup and redirect parent.
     */
    protected function closePopupAndRedirect($url)
    {
        $content = "<html><head><script>
            window.opener.location.href = '$url';
            window.close();
        </script></head><body></body></html>";
        return new \Symfony\Component\HttpFoundation\Response($content);
    }

    /**
     * Query installment options via AJAX.
     *
     * @param \Symfony\Component\HttpFoundation\Request $request
     *   The request.
     *
     * @return \Symfony\Component\HttpFoundation\JsonResponse
     *   A JSON response.
     */
    public function queryInstallments(Request $request)
    {
        $bin = $request->query->get('bin');
        $amount = $request->query->get('amount');
        $merchant_id = $request->query->get('merchant_id');
        $merchant_user = $request->query->get('merchant_user');
        $merchant_password = $request->query->get('merchant_password');
        $test_mode = $request->query->get('test_mode', TRUE);

        if (empty($bin) || empty($amount) || empty($merchant_id)) {
            return new JsonResponse(['error' => 'Missing required parameters'], 400);
        }

        $response = $this->apiClient->queryInstallments([
            'MERCHANT' => $merchant_id,
            'MERCHANTUSER' => $merchant_user,
            'MERCHANTPASSWORD' => $merchant_password,
            'BIN' => $bin,
            'AMOUNT' => $amount,
            'test_mode' => $test_mode,
        ]);

        if (empty($response) || $response['responseCode'] !== '00') {
            return new JsonResponse([
                'installments' => [
                    ['count' => 1, 'label' => $this->t('Single payment')],
                ],
            ]);
        }

        // Parse installment options from response.
        $installments = [];
        $installments[] = ['count' => 1, 'label' => $this->t('Single payment')];

        if (!empty($response['installmentList'])) {
            foreach ($response['installmentList'] as $installment) {
                if ($installment['installmentCount'] > 1) {
                    $installments[] = [
                        'count' => $installment['installmentCount'],
                        'label' => $this->t('@count installments', ['@count' => $installment['installmentCount']]),
                    ];
                }
            }
        }

        return new JsonResponse(['installments' => $installments]);
    }
}
