/**
 * @file
 * Paratika payment form JavaScript.
 */

(function ($, Drupal, drupalSettings) {
  "use strict";

  Drupal.behaviors.paratikaPayment = {
    attach: function (context, settings) {
      // 1. Handle Paratika Form internals (formatting, etc).
      // This only runs when the Paratika form itself is loaded/refreshed.
      var $container = $("#paratika-payment-form", context);

      if ($container.length > 0 && !$container.hasClass("paratika-processed")) {
        $container.addClass("paratika-processed");

        // Find the actual form element (parent of the container)
        var $form = $container.closest("form");

        var paratikaSettings = drupalSettings.commerceParatika || {};
        var $cardNumber = $(".paratika-card-number", $container);
        var $installments = $(".paratika-installments", $container);
        var currentBin = "";

        // Handle form submission.
        $form.on("submit", function (e) {
          // Check if the trigger is the Paratika payment
          if ($container.is(":hidden")) {
            return;
          }

          e.preventDefault();

          var $submitButton = $form.find(
            'input[type="submit"], button[type="submit"]'
          );
          $submitButton.prop("disabled", true);

          // Prepare payment data.
          var cardNumber = $cardNumber.val().replace(/\s/g, "");
          var cardHolder = $(".paratika-card-holder", $form).val();
          var expiryMonth = $(".paratika-expiry-month", $form).val();
          var expiryYear = $(".paratika-expiry-year", $form).val();
          var cvv = $(".paratika-cvv", $form).val();
          var installments = $installments.val();
          var sessionToken = paratikaSettings.sessionToken;

          // Create a hidden form to post to Paratika.
          var $paratikaForm = $("<form></form>")
            .attr("method", "POST")
            .attr("action", paratikaSettings.saleUrl)
            //.attr('target', 'paratika-3ds-frame') // Removed target to redirect current page
            .hide();

          // Add form fields.
          $paratikaForm.append(createHiddenField("ACTION", "SALE"));
          $paratikaForm.append(createHiddenField("sessionToken", sessionToken));
          $paratikaForm.append(createHiddenField("pan", cardNumber));
          $paratikaForm.append(createHiddenField("expiryMonth", expiryMonth));
          $paratikaForm.append(createHiddenField("expiryYear", expiryYear));
          $paratikaForm.append(createHiddenField("cvv", cvv));
          $paratikaForm.append(createHiddenField("cardOwner", cardHolder));
          $paratikaForm.append(
            createHiddenField("installmentCount", installments)
          );

          // Append form to body.
          $("body").append($paratikaForm);

          // Submit the form based on method.
          if (paratikaSettings.method3ds === "popup") {
            var popupName = "paratika_3ds_popup";
            var popup = window.open(
              "about:blank",
              popupName,
              "width=800,height=600,scrollbars=yes,resizable=yes"
            );

            if (popup) {
              $paratikaForm.attr("target", popupName);
              $paratikaForm.submit();
            } else {
              alert(
                Drupal.t(
                  "Please enable popups for this site to complete payment."
                )
              );
            }
          } else {
            // Redirect (default).
            $paratikaForm.submit();
          }
        });

        // Update installment select options.
        function updateInstallmentOptions(installments) {
          $installments.empty();
          $.each(installments, function (index, installment) {
            $installments.append(
              $("<option></option>")
                .attr("value", installment.count)
                .text(installment.label)
            );
          });

          // Hide installments if only single payment is available.
          var $wrapper = $("#paratika-installments-wrapper", $container);
          if (installments.length <= 1) {
            $wrapper.hide();
          } else {
            $wrapper.show();
          }
        }

        // Initial check for installments visibility (default is 1 option)
        var $wrapper = $("#paratika-installments-wrapper", $container);
        if ($installments.children("option").length <= 1) {
          $wrapper.hide();
        }

        // Only query installments if enabled.
        if (paratikaSettings.enableInstallments) {
          // Listen for bin changes (first 6 digits).
          $cardNumber.on("input", function () {
            var bin = $(this).val().replace(/\D/g, "").substring(0, 6);
            if (bin.length === 6 && bin !== currentBin) {
              currentBin = bin;

              // Query installments.
              $.ajax({
                url: paratikaSettings.installmentsUrl,
                data: {
                  bin: bin,
                  amount: paratikaSettings.amount,
                  merchant_id: paratikaSettings.merchantId, // Note: Controller needs to handle this if not passed securely
                  test_mode: paratikaSettings.testMode,
                },
                success: function (response) {
                  if (response.installments) {
                    updateInstallmentOptions(response.installments);
                  }
                },
              });
            }
          });
        } else {
          // Hide installments wrapper if disabled.
          $wrapper.hide();
        }
      }

      // 2. Handle Visibility Toggle.
      // This needs to run whenever the payment method radios are present in the context,
      // OR when the Paratika form is present (to set initial state).
      var $globalContainer = $("#paratika-payment-form");
      if ($globalContainer.length > 0) {
        var $form = $globalContainer.closest("form");

        var toggleVisibility = function () {
          var selected = $(
            'input[name="payment_information[payment_method]"]:checked',
            $form
          ).val();
          if (selected && selected.indexOf("paratika") === 0) {
            $globalContainer.show();
            // Re-enable required validation when showing the form
            $globalContainer
              .find(
                'input[data-was-required="true"], select[data-was-required="true"]'
              )
              .each(function () {
                $(this).prop("required", true);
              });
          } else {
            $globalContainer.hide();
            // Disable required validation when hiding the form to prevent HTML5 validation errors
            $globalContainer
              .find("input[required], select[required]")
              .each(function () {
                $(this).attr("data-was-required", "true");
                $(this).prop("required", false);
              });
          }
        };

        // Bind to radios in the current context (e.g. when payment info pane refreshes).
        var $radios = $(
          'input[name="payment_information[payment_method]"]',
          context
        );
        if ($radios.length > 0) {
          $radios.on("change", toggleVisibility);
          // Run once to ensure correct state if radios just loaded.
          toggleVisibility();
        }

        // Also run toggle if we just loaded the container (and radios are already there).
        if ($container.length > 0) {
          toggleVisibility();
        }
      }

      // Create hidden input field helper.
      function createHiddenField(name, value) {
        return $("<input>")
          .attr("type", "hidden")
          .attr("name", name)
          .val(value);
      }
    },
  };
})(jQuery, Drupal, drupalSettings);
