<?php

namespace Drupal\commerce_eft_payment\Plugin\Commerce\PaymentGateway;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\commerce_payment\Attribute\CommercePaymentGateway;
use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\ManualPaymentGatewayInterface;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\PaymentGatewayBase;
use Drupal\commerce_payment\PluginForm\ManualPaymentAddForm;
use Drupal\commerce_payment\PluginForm\PaymentReceiveForm;
use Drupal\commerce_price\Price;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides the EFT/Bank Transfer payment gateway.
 */
#[CommercePaymentGateway(
    id: "eft",
    label: new TranslatableMarkup("EFT/Bank Transfer"),
    display_label: new TranslatableMarkup("EFT/Havale"),
    modes: [
        "n/a" => new TranslatableMarkup("N/A"),
    ],
    forms: [
        "add-payment" => ManualPaymentAddForm::class,
        "receive-payment" => PaymentReceiveForm::class,
    ],
    payment_type: "payment_manual",
    requires_billing_information: FALSE,
)]
class Eft extends PaymentGatewayBase implements ManualPaymentGatewayInterface
{

    /**
     * The token service.
     *
     * @var \Drupal\Core\Utility\Token
     */
    protected $token;

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
    {
        $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
        $instance->token = $container->get('token');
        return $instance;
    }

    /**
     * {@inheritdoc}
     */
    public function defaultConfiguration()
    {
        return [
            'instructions' => [
                'value' => '',
                'format' => 'plain_text',
            ],
            'show_order_currency_only' => FALSE,
            'bank_accounts' => [],
        ] + parent::defaultConfiguration();
    }

    /**
     * Returns the available currency options.
     */
    protected function getCurrencyOptions(): array
    {
        return [
            'TRY' => $this->t('TRY - Turkish Lira'),
            'USD' => $this->t('USD - US Dollar'),
            'EUR' => $this->t('EUR - Euro'),
            'GBP' => $this->t('GBP - British Pound'),
            'CHF' => $this->t('CHF - Swiss Franc'),
            'JPY' => $this->t('JPY - Japanese Yen'),
            'CAD' => $this->t('CAD - Canadian Dollar'),
            'AUD' => $this->t('AUD - Australian Dollar'),
            'CNY' => $this->t('CNY - Chinese Yuan'),
            'RUB' => $this->t('RUB - Russian Ruble'),
            'SAR' => $this->t('SAR - Saudi Riyal'),
            'AED' => $this->t('AED - UAE Dirham'),
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function buildConfigurationForm(array $form, FormStateInterface $form_state)
    {
        $form = parent::buildConfigurationForm($form, $form_state);

        $form['instructions'] = [
            '#type' => 'text_format',
            '#title' => $this->t('Payment instructions'),
            '#description' => $this->t('Shown the end of checkout, after the customer has placed their order.'),
            '#default_value' => $this->configuration['instructions']['value'],
            '#format' => $this->configuration['instructions']['format'],
        ];

        $form['show_order_currency_only'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Show only order currency accounts'),
            '#description' => $this->t('When enabled, only bank accounts matching the order currency will be shown to customers.'),
            '#default_value' => $this->configuration['show_order_currency_only'] ?? FALSE,
        ];

        // Get the bank accounts from form storage or configuration.
        // Using storage instead of set/get for better AJAX persistence.
        $storage = $form_state->getStorage();
        if (!isset($storage['bank_accounts'])) {
            // Initialize from configuration.
            $storage['bank_accounts'] = $this->configuration['bank_accounts'] ?: [];
            $form_state->setStorage($storage);
        }
        $bank_accounts = $storage['bank_accounts'];

        // Bank accounts fieldset.
        $form['bank_accounts_fieldset'] = [
            '#type' => 'fieldset',
            '#title' => $this->t('Bank Accounts'),
            '#prefix' => '<div id="eft-bank-accounts-wrapper">',
            '#suffix' => '</div>',
        ];

        // Build the bank accounts table.
        $form['bank_accounts_fieldset']['accounts'] = [
            '#type' => 'table',
            '#header' => [
                $this->t('Enabled'),
                $this->t('Currency'),
                $this->t('Bank Name'),
                $this->t('Recipient Name'),
                $this->t('IBAN'),
                $this->t('Operations'),
            ],
            '#empty' => $this->t('No bank accounts configured. Click "Add bank account" to add one.'),
            '#prefix' => '<div id="bank-accounts-table">',
            '#suffix' => '</div>',
        ];

        foreach ($bank_accounts as $index => $account) {
            $form['bank_accounts_fieldset']['accounts'][$index]['enabled'] = [
                '#type' => 'checkbox',
                '#default_value' => $account['enabled'] ?? TRUE,
            ];
            $form['bank_accounts_fieldset']['accounts'][$index]['currency'] = [
                '#type' => 'select',
                '#options' => $this->getCurrencyOptions(),
                '#default_value' => $account['currency'] ?? 'TRY',
            ];
            $form['bank_accounts_fieldset']['accounts'][$index]['bank_name'] = [
                '#type' => 'textfield',
                '#default_value' => $account['bank_name'] ?? '',
                '#size' => 20,
            ];
            $form['bank_accounts_fieldset']['accounts'][$index]['recipient_name'] = [
                '#type' => 'textfield',
                '#default_value' => $account['recipient_name'] ?? '',
                '#size' => 25,
            ];
            $form['bank_accounts_fieldset']['accounts'][$index]['iban'] = [
                '#type' => 'textfield',
                '#default_value' => $account['iban'] ?? '',
                '#size' => 30,
            ];
            $form['bank_accounts_fieldset']['accounts'][$index]['remove'] = [
                '#type' => 'submit',
                '#value' => $this->t('Remove'),
                '#name' => 'remove_bank_account_' . $index,
                '#submit' => [[$this, 'rebuildFormSubmit']],
                '#ajax' => [
                    'callback' => [static::class, 'ajaxCallback'],
                    'wrapper' => 'eft-bank-accounts-wrapper',
                ],
                '#limit_validation_errors' => [],
            ];
        }

        $form['bank_accounts_fieldset']['add_account'] = [
            '#type' => 'submit',
            '#value' => $this->t('Add bank account'),
            '#name' => 'add_bank_account',
            '#submit' => [[$this, 'rebuildFormSubmit']],
            '#ajax' => [
                'callback' => [static::class, 'ajaxCallback'],
                'wrapper' => 'eft-bank-accounts-wrapper',
            ],
            '#limit_validation_errors' => [],
        ];

        return $form;
    }

    /**
     * AJAX callback for bank account add/remove buttons.
     */
    public static function ajaxCallback(array &$form, FormStateInterface $form_state)
    {
        // Get the triggering element to find the correct form path.
        $triggering_element = $form_state->getTriggeringElement();

        // Navigate up from the triggering element to find bank_accounts_fieldset.
        // The triggering element's parents tell us where it is in the form.
        $parents = $triggering_element['#array_parents'];

        // Find the bank_accounts_fieldset by traversing up.
        $element = $form;
        foreach ($parents as $parent) {
            if ($parent === 'bank_accounts_fieldset') {
                return $element['bank_accounts_fieldset'];
            }
            if (isset($element[$parent])) {
                $element = $element[$parent];
            }
        }

        // Fallback: try common paths.
        if (isset($form['configuration']['bank_accounts_fieldset'])) {
            return $form['configuration']['bank_accounts_fieldset'];
        }
        if (isset($form['bank_accounts_fieldset'])) {
            return $form['bank_accounts_fieldset'];
        }

        // Last resort: return the whole form.
        return $form;
    }

    /**
     * Submit callback to rebuild the form.
     */
    public function rebuildFormSubmit(array &$form, FormStateInterface $form_state)
    {
        // Get current bank accounts from various sources.
        $bank_accounts = [];

        // First, try to get from form_state values (various possible paths).
        $values = $form_state->getValue(['bank_accounts_fieldset', 'accounts'])
            ?: $form_state->getValue(['configuration', 'bank_accounts_fieldset', 'accounts'])
            ?: $form_state->getValue(['configuration', 'eft', 'bank_accounts_fieldset', 'accounts'])
            ?: [];

        // If we got values from form state, use them.
        if (!empty($values)) {
            foreach ($values as $account) {
                if (is_array($account) && isset($account['currency'])) {
                    $bank_accounts[] = [
                        'enabled' => !empty($account['enabled']),
                        'currency' => $account['currency'] ?? 'TRY',
                        'bank_name' => $account['bank_name'] ?? '',
                        'recipient_name' => $account['recipient_name'] ?? '',
                        'iban' => $account['iban'] ?? '',
                    ];
                }
            }
        }

        // If still empty, try getting from form storage.
        if (empty($bank_accounts)) {
            $storage = $form_state->getStorage();
            if (!empty($storage['bank_accounts'])) {
                $bank_accounts = $storage['bank_accounts'];
            }
        }

        // If still empty, try getting from the form array default values.
        if (empty($bank_accounts) && isset($form['bank_accounts_fieldset']['accounts'])) {
            $accounts_table = $form['bank_accounts_fieldset']['accounts'];
            foreach ($accounts_table as $key => $row) {
                if (is_numeric($key) && isset($row['currency']['#default_value'])) {
                    $bank_accounts[] = [
                        'enabled' => !empty($row['enabled']['#default_value']),
                        'currency' => $row['currency']['#default_value'] ?? 'TRY',
                        'bank_name' => $row['bank_name']['#default_value'] ?? '',
                        'recipient_name' => $row['recipient_name']['#default_value'] ?? '',
                        'iban' => $row['iban']['#default_value'] ?? '',
                    ];
                }
            }
        }

        // Handle add/remove based on triggering element.
        $triggering_element = $form_state->getTriggeringElement();
        $op = $triggering_element['#name'] ?? '';

        if ($op === 'add_bank_account') {
            $bank_accounts[] = [
                'enabled' => TRUE,
                'currency' => 'TRY',
                'bank_name' => '',
                'recipient_name' => '',
                'iban' => '',
            ];
        } elseif (preg_match('/^remove_bank_account_(\d+)$/', $op, $matches)) {
            $index = (int) $matches[1];
            if (isset($bank_accounts[$index])) {
                unset($bank_accounts[$index]);
                $bank_accounts = array_values($bank_accounts);
            }
        }

        // Use storage for persistence across AJAX.
        $storage = $form_state->getStorage();
        $storage['bank_accounts'] = $bank_accounts;
        $form_state->setStorage($storage);
        $form_state->setRebuild();
    }

    /**
     * {@inheritdoc}
     */
    public function submitConfigurationForm(array &$form, FormStateInterface $form_state)
    {
        parent::submitConfigurationForm($form, $form_state);

        if (!$form_state->getErrors()) {
            $values = $form_state->getValue($form['#parents']);
            $this->configuration['instructions'] = $values['instructions'];
            $this->configuration['show_order_currency_only'] = !empty($values['show_order_currency_only']);

            // Save bank accounts from the table.
            $bank_accounts = [];
            if (!empty($values['bank_accounts_fieldset']['accounts'])) {
                foreach ($values['bank_accounts_fieldset']['accounts'] as $account) {
                    if (is_array($account) && (!empty($account['bank_name']) || !empty($account['iban']))) {
                        $bank_accounts[] = [
                            'enabled' => !empty($account['enabled']),
                            'currency' => $account['currency'] ?? 'TRY',
                            'bank_name' => $account['bank_name'] ?? '',
                            'recipient_name' => $account['recipient_name'] ?? '',
                            'iban' => $account['iban'] ?? '',
                        ];
                    }
                }
            }
            $this->configuration['bank_accounts'] = $bank_accounts;
        }
    }

    /**
     * {@inheritdoc}
     */
    public function buildPaymentInstructions(PaymentInterface $payment)
    {
        $build = [
            '#attached' => [
                'library' => ['commerce_eft_payment/eft-payment'],
            ],
        ];

        // Add text instructions if set.
        if (!empty($this->configuration['instructions']['value'])) {
            $instructions_text = $this->token->replace($this->configuration['instructions']['value'], [
                'commerce_order' => $payment->getOrder(),
                'commerce_payment' => $payment,
            ]);

            $build['instructions'] = [
                '#type' => 'processed_text',
                '#text' => $instructions_text,
                '#format' => $this->configuration['instructions']['format'],
            ];
        }

        // Add bank accounts table (only enabled ones).
        $enabled_accounts = array_filter($this->configuration['bank_accounts'] ?? [], function ($account) {
            return !empty($account['enabled']);
        });

        // Filter by order currency if enabled.
        if (!empty($this->configuration['show_order_currency_only'])) {
            $order = $payment->getOrder();
            $order_currency = $order->getTotalPrice()->getCurrencyCode();

            $enabled_accounts = array_filter($enabled_accounts, function ($account) use ($order_currency) {
                return ($account['currency'] ?? '') === $order_currency;
            });
        }

        if (!empty($enabled_accounts)) {
            $build['bank_accounts'] = [
                '#type' => 'table',
                '#caption' => $this->t('Bank Account Information'),
                '#header' => [
                    $this->t('Currency'),
                    $this->t('Bank Name'),
                    $this->t('Recipient Name'),
                    $this->t('IBAN'),
                ],
                '#attributes' => [
                    'class' => ['eft-bank-accounts-table'],
                ],
            ];

            foreach ($enabled_accounts as $index => $account) {
                $build['bank_accounts'][$index] = [
                    'currency' => ['#markup' => $account['currency']],
                    'bank_name' => ['#markup' => $account['bank_name']],
                    'recipient_name' => ['#markup' => $account['recipient_name']],
                    'iban' => ['#markup' => $account['iban']],
                ];
            }
        }

        return $build;
    }

    /**
     * {@inheritdoc}
     */
    public function buildPaymentOperations(PaymentInterface $payment)
    {
        $operations = [];
        $operations['receive'] = [
            'title' => $this->t('Receive'),
            'page_title' => $this->t('Receive payment'),
            'plugin_form' => 'receive-payment',
            'access' => $payment->getState()->getId() === 'pending',
        ];
        $operations['void'] = [
            'title' => $this->t('Void'),
            'page_title' => $this->t('Void payment'),
            'plugin_form' => 'void-payment',
            'access' => $this->canVoidPayment($payment),
        ];
        $operations['refund'] = [
            'title' => $this->t('Refund'),
            'page_title' => $this->t('Refund payment'),
            'plugin_form' => 'refund-payment',
            'access' => $this->canRefundPayment($payment),
        ];

        return $operations;
    }

    /**
     * {@inheritdoc}
     */
    public function canVoidPayment(PaymentInterface $payment)
    {
        return $payment->getState()->getId() === 'pending';
    }

    /**
     * {@inheritdoc}
     */
    public function createPayment(PaymentInterface $payment, $received = FALSE)
    {
        $this->assertPaymentState($payment, ['new']);

        $payment->state = $received ? 'completed' : 'pending';
        $payment->save();
    }

    /**
     * {@inheritdoc}
     */
    public function receivePayment(PaymentInterface $payment, ?Price $amount = NULL)
    {
        $this->assertPaymentState($payment, ['pending']);

        // If not specified, use the entire amount.
        $amount = $amount ?: $payment->getAmount();
        $payment->state = 'completed';
        $payment->setAmount($amount);
        $payment->save();
    }

    /**
     * {@inheritdoc}
     */
    public function voidPayment(PaymentInterface $payment)
    {
        $this->assertPaymentState($payment, ['pending']);

        $payment->state = 'voided';
        $payment->save();
    }

    /**
     * {@inheritdoc}
     */
    public function refundPayment(PaymentInterface $payment, ?Price $amount = NULL)
    {
        $this->assertPaymentState($payment, ['completed', 'partially_refunded']);
        // If not specified, refund the entire amount.
        $amount = $amount ?: $payment->getAmount();
        $this->assertRefundAmount($payment, $amount);

        $old_refunded_amount = $payment->getRefundedAmount();
        $new_refunded_amount = $old_refunded_amount->add($amount);
        if ($new_refunded_amount->lessThan($payment->getAmount())) {
            $payment->state = 'partially_refunded';
        } else {
            $payment->state = 'refunded';
        }

        $payment->setRefundedAmount($new_refunded_amount);
        $payment->save();
    }

    /**
     * Gets the configured bank accounts.
     *
     * @return array
     *   Array of bank account configurations.
     */
    public function getBankAccounts()
    {
        return $this->configuration['bank_accounts'] ?? [];
    }
}
