<?php

namespace Drupal\Tests\commerce_courier_shipping\Unit\Service;

use Drupal\commerce_courier_shipping\Service\TableRatesService;
use Drupal\Tests\UnitTestCase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\address_tr\Service\SubdivisionLoaderService;
use Psr\Log\LoggerInterface;
use Drupal\commerce_price\Price;

/**
 * Tests for TableRatesService.
 *
 * @group commerce_courier_shipping
 * @coversDefaultClass \Drupal\commerce_courier_shipping\Service\TableRatesService
 */
class TableRatesServiceTest extends UnitTestCase
{

    /**
     * @var \Drupal\commerce_courier_shipping\Service\TableRatesService
     */
    protected $tableRatesService;

    /**
     * {@inheritdoc}
     */
    protected function setUp(): void
    {
        parent::setUp();

        $configFactory = $this->createMock(ConfigFactoryInterface::class);
        $subdivisionLoader = $this->createMock(SubdivisionLoaderService::class);
        $logger = $this->createMock(LoggerInterface::class);

        $this->tableRatesService = new TableRatesService($configFactory, $subdivisionLoader, $logger);
    }

    /**
     * Tests finding a specific neighbourhood rate.
     */
    public function testFindRateNeighbourhoodMatch()
    {
        $rates = [
            [
                'province_code' => '34',
                'district_code' => 'KADIKOY',
                'neighbourhood_code' => 'MODA',
                'price' => '10.00',
                'currency_code' => 'TRY',
            ],
            [
                'province_code' => '34', // General Istanbul rate
                'price' => '20.00',
                'currency_code' => 'TRY',
            ]
        ];

        $price = $this->tableRatesService->findRate($rates, '34', 'KADIKOY', 'MODA');
        $this->assertNotNull($price);
        $this->assertEquals('10.00', $price->getNumber());
    }

    /**
     * Tests fallback to district rate.
     */
    public function testFindRateDistrictFallback()
    {
        $rates = [
            [
                'province_code' => '34',
                'district_code' => 'KADIKOY',
                'price' => '15.00',
                'currency_code' => 'TRY',
            ],
            [
                'province_code' => '34',
                'price' => '20.00',
                'currency_code' => 'TRY',
            ]
        ];

        // Searching for a neighbourhood NOT in the rates, but district is there.
        $price = $this->tableRatesService->findRate($rates, '34', 'KADIKOY', 'UNKNOWN_NEIGHBOURHOOD');
        $this->assertNotNull($price);
        $this->assertEquals('15.00', $price->getNumber());
    }

    /**
     * Tests fallback to city (province) rate.
     */
    public function testFindRateCityFallback()
    {
        $rates = [
            [
                'province_code' => '34',
                'price' => '25.00',
                'currency_code' => 'TRY',
            ]
        ];

        $price = $this->tableRatesService->findRate($rates, '34', 'SISLI', 'CENTRAL');
        $this->assertNotNull($price);
        $this->assertEquals('25.00', $price->getNumber());
    }

    /**
     * Tests no match found.
     */
    public function testFindRateNoMatch()
    {
        $rates = [
            [
                'province_code' => '34',
                'price' => '25.00',
                'currency_code' => 'TRY',
            ]
        ];

        // Searching for Ankara (06), only Istanbul (34) is defined.
        $price = $this->tableRatesService->findRate($rates, '06', 'CANKAYA', '');
        $this->assertNull($price);
    }
}
