<?php

namespace Drupal\Tests\commerce_courier_shipping\Unit\Service;

use Drupal\commerce_courier_shipping\Service\PriceModifierService;
use Drupal\Tests\UnitTestCase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Component\Datetime\TimeInterface;
use Psr\Log\LoggerInterface;

/**
 * Tests for PriceModifierService.
 *
 * @group commerce_courier_shipping
 * @coversDefaultClass \Drupal\commerce_courier_shipping\Service\PriceModifierService
 */
class PriceModifierServiceTest extends UnitTestCase
{

    /**
     * @var \Drupal\commerce_courier_shipping\Service\PriceModifierService
     */
    protected $priceModifierService;

    /**
     * {@inheritdoc}
     */
    protected function setUp(): void
    {
        parent::setUp();

        $configFactory = $this->createMock(ConfigFactoryInterface::class);
        $time = $this->createMock(TimeInterface::class);
        $logger = $this->createMock(LoggerInterface::class);

        $this->priceModifierService = new PriceModifierService($configFactory, $time, $logger);
    }

    /**
     * Tests daily multiplier (e.g., weekends).
     */
    public function testDailyMultiplier()
    {
        // Monday = 1, Sunday = 7.
        // Let's say Sunday is 1.5x.
        $daily_modifiers = [
            [
                'days' => [7],
                'multiplier' => '1.5',
            ]
        ];

        // Test on a Sunday.
        $sunday = new \DateTime('2023-10-01'); // Oct 1, 2023 is Sunday.
        $this->assertEquals(1.5, $this->priceModifierService->calculateMultiplier($sunday, $daily_modifiers, [], []));

        // Test on a Monday.
        $monday = new \DateTime('2023-10-02');
        $this->assertEquals(1.0, $this->priceModifierService->calculateMultiplier($monday, $daily_modifiers, [], []));
    }

    /**
     * Tests hourly multiplier (e.g., peak hours).
     */
    public function testHourlyMultiplier()
    {
        // Peak hours: 18:00 - 20:00 is 1.2x.
        $hour_modifiers = [
            [
                'start_hour' => '18:00',
                'end_hour' => '20:00',
                'multiplier' => '1.2',
            ]
        ];

        // 19:00 - Inside.
        $time_in = new \DateTime('2023-10-02 19:00:00');
        $this->assertEquals(1.2, $this->priceModifierService->calculateMultiplier($time_in, [], [], $hour_modifiers));

        // 17:00 - Outside.
        $time_out = new \DateTime('2023-10-02 17:00:00');
        $this->assertEquals(1.0, $this->priceModifierService->calculateMultiplier($time_out, [], [], $hour_modifiers));
    }

    /**
     * Tests overnight hourly multiplier.
     */
    public function testOvernightHourlyMultiplier()
    {
        // Overnight: 22:00 - 06:00 is 2.0x.
        $hour_modifiers = [
            [
                'start_hour' => '22:00',
                'end_hour' => '06:00',
                'multiplier' => '2.0',
            ]
        ];

        // 23:00 - Inside (late night).
        $late_night = new \DateTime('2023-10-02 23:00:00');
        $this->assertEquals(2.0, $this->priceModifierService->calculateMultiplier($late_night, [], [], $hour_modifiers));

        // 04:00 - Inside (early morning).
        $early_morning = new \DateTime('2023-10-03 04:00:00');
        $this->assertEquals(2.0, $this->priceModifierService->calculateMultiplier($early_morning, [], [], $hour_modifiers));

        // 12:00 - Outside (noon).
        $noon = new \DateTime('2023-10-03 12:00:00');
        $this->assertEquals(1.0, $this->priceModifierService->calculateMultiplier($noon, [], [], $hour_modifiers));
    }

    /**
     * Tests combined multipliers (stacking).
     */
    public function testCombinedMultipliers()
    {
        // Sunday (1.5x) AND 8:00-10:00 (1.1x).
        $daily_modifiers = [
            ['days' => [7], 'multiplier' => '1.5']
        ];
        $hour_modifiers = [
            ['start_hour' => '08:00', 'end_hour' => '10:00', 'multiplier' => '1.1']
        ];

        // Sunday at 09:00.
        $datetime = new \DateTime('2023-10-01 09:00:00');

        // Expected: 1.5 * 1.1 = 1.65.
        $result = $this->priceModifierService->calculateMultiplier($datetime, $daily_modifiers, [], $hour_modifiers);
        $this->assertEquals(1.65, $result);
    }
}
