<?php

namespace Drupal\Tests\commerce_courier_shipping\Unit\Service;

use Drupal\commerce_courier_shipping\Service\LocationMatcherService;
use Drupal\Tests\UnitTestCase;
use Psr\Log\LoggerInterface;

/**
 * Tests for LocationMatcherService.
 *
 * @group commerce_courier_shipping
 * @coversDefaultClass \Drupal\commerce_courier_shipping\Service\LocationMatcherService
 */
class LocationMatcherServiceTest extends UnitTestCase
{

    /**
     * @var \Drupal\commerce_courier_shipping\Service\LocationMatcherService
     */
    protected $locationMatcher;

    /**
     * {@inheritdoc}
     */
    protected function setUp(): void
    {
        parent::setUp();
        $logger = $this->createMock(LoggerInterface::class);
        $this->locationMatcher = new LocationMatcherService($logger);
    }

    /**
     * Tests behavior when no locations are enabled (should allow all).
     */
    public function testEmptyConfigAllowsAll()
    {
        $enabled_locations = [];
        $result = $this->locationMatcher->isLocationEnabled($enabled_locations, '34', 'KADIKOY');
        $this->assertTrue($result, 'Empty config should allow all locations.');
    }

    /**
     * Tests exact match for province and district.
     */
    public function testExactDistrictMatch()
    {
        $enabled_locations = [
            [
                'province' => '34',
                'districts' => ['KADIKOY'],
            ]
        ];

        $this->assertTrue($this->locationMatcher->isLocationEnabled($enabled_locations, '34', 'KADIKOY'));
        $this->assertFalse($this->locationMatcher->isLocationEnabled($enabled_locations, '34', 'BETIKS'));
        $this->assertFalse($this->locationMatcher->isLocationEnabled($enabled_locations, '06', 'KADIKOY'));
    }

    /**
     * Tests "All Districts" wildcard for a province.
     */
    public function testAllDistrictsWildcard()
    {
        $enabled_locations = [
            [
                'province' => '06', // Ankara
                'all_districts' => TRUE,
            ]
        ];

        $this->assertTrue($this->locationMatcher->isLocationEnabled($enabled_locations, '06', 'CANKAYA'));
        $this->assertTrue($this->locationMatcher->isLocationEnabled($enabled_locations, '06', 'KECIOREN'));
        $this->assertFalse($this->locationMatcher->isLocationEnabled($enabled_locations, '34', 'KADIKOY'));
    }

    /**
     * Tests neighbourhood restrictions.
     */
    public function testNeighbourhoodMatch()
    {
        $enabled_locations = [
            [
                'province' => '34',
                'districts' => ['KADIKOY'],
                'neighbourhoods' => ['MODA', 'CAFERAGA'],
            ]
        ];

        $this->assertTrue($this->locationMatcher->isLocationEnabled($enabled_locations, '34', 'KADIKOY', 'MODA'));
        $this->assertTrue($this->locationMatcher->isLocationEnabled($enabled_locations, '34', 'KADIKOY', 'CAFERAGA'));
        // Fallback: If neighbourhoods are defined, other neighbourhoods in same district should be blocked?
        // Note: The logic in LocationMatcherService says:
        // if (in_array($neighbourhood_code, $neighbourhoods)) return TRUE;
        // It doesn't explicitly return FALSE immediately if district matches but neighbourhood doesn't, 
        // it continues loop. But since this is the only entry, it will eventually return FALSE.
        $this->assertFalse($this->locationMatcher->isLocationEnabled($enabled_locations, '34', 'KADIKOY', 'UNKNOWN'));
    }

    /**
     * Tests "All Neighbourhoods" for a district.
     */
    public function testAllNeighbourhoodsWildcard()
    {
        $enabled_locations = [
            [
                'province' => '34',
                'districts' => ['KADIKOY'],
                'all_district_neighbourhoods' => ['KADIKOY'], // "All" selected for Kadikoy
            ]
        ];

        $this->assertTrue($this->locationMatcher->isLocationEnabled($enabled_locations, '34', 'KADIKOY', 'ANYTHING'));
    }
}
