<?php

namespace Drupal\commerce_courier_shipping\Service;

use Drupal\commerce_price\Price;

/**
 * Interface for the table rates service.
 */
interface TableRatesServiceInterface
{

    /**
     * Finds the rate for a given address from a provided list of rates.
     *
     * Returns the most specific matching rate (neighbourhood > district > city).
     *
     * @param array $rates
     *   The rates array to search in.
     * @param string $province_code
     *   The province code (e.g., TR-34).
     * @param string $district_code
     *   The district code (e.g., KADIKOY).
     * @param string $neighbourhood_code
     *   The neighbourhood code (e.g., CAFERAGA-MAHALLESI).
     *
     * @return \Drupal\commerce_price\Price|null
     *   The matching rate, or NULL if no rate found.
     */
    public function findRate(array $rates, string $province_code, string $district_code, string $neighbourhood_code): ?Price;

    /**
     * Gets the rate for a given address.
     *
     * Returns the most specific matching rate (neighbourhood > district > city).
     *
     * @param string $province_code
     *   The province code (e.g., TR-34).
     * @param string $district_code
     *   The district code (e.g., KADIKOY).
     * @param string $neighbourhood_code
     *   The neighbourhood code (e.g., CAFERAGA-MAHALLESI).
     *
     * @return \Drupal\commerce_price\Price|null
     *   The matching rate, or NULL if no rate found.
     */
    public function getRateForAddress(string $province_code, string $district_code, string $neighbourhood_code): ?Price;

    /**
     * Gets all table rates.
     *
     * @return array
     *   Array of table rate entries.
     */
    public function getAllRates(): array;

    /**
     * Saves table rates.
     *
     * @param array $rates
     *   Array of table rate entries.
     */
    public function saveRates(array $rates): void;

    /**
     * Imports table rates from CSV content.
     *
     * @param string $csv_content
     *   The CSV content.
     *
     * @return array
     *   Array with 'success' (bool), 'imported' (int), 'errors' (array).
     */
    public function importFromCsv(string $csv_content): array;

    /**
     * Exports table rates to CSV format.
     *
     * @return string
     *   The CSV content.
     */
    public function exportToCsv(): string;

    /**
     * Gets province options for form selects.
     *
     * @return array
     *   Array of province options keyed by code.
     */
    public function getProvinceOptions(): array;

    /**
     * Gets district options for a province.
     *
     * @param string $province_code
     *   The province code.
     *
     * @return array
     *   Array of district options keyed by code.
     */
    public function getDistrictOptions(string $province_code): array;

    /**
     * Gets neighbourhood options for a district.
     *
     * @param string $province_code
     *   The province code.
     * @param string $district_code
     *   The district code.
     *
     * @return array
     *   Array of neighbourhood options keyed by code.
     */
    public function getNeighbourhoodOptions(string $province_code, string $district_code): array;
}
