<?php

namespace Drupal\commerce_courier_shipping\Service;

use Drupal\address_tr\Service\SubdivisionLoaderService;
use Drupal\commerce_price\Price;
use Drupal\Core\Config\ConfigFactoryInterface;
use Psr\Log\LoggerInterface;

/**
 * Service for managing table rates.
 */
class TableRatesService implements TableRatesServiceInterface
{

    /**
     * The config factory.
     *
     * @var \Drupal\Core\Config\ConfigFactoryInterface
     */
    protected ConfigFactoryInterface $configFactory;

    /**
     * The subdivision loader service.
     *
     * @var \Drupal\address_tr\Service\SubdivisionLoaderService
     */
    protected SubdivisionLoaderService $subdivisionLoader;

    /**
     * The logger.
     *
     * @var \Psr\Log\LoggerInterface
     */
    protected LoggerInterface $logger;

    /**
     * Cache for subdivision data.
     *
     * @var array
     */
    protected array $subdivisionCache = [];

    /**
     * Constructs a TableRatesService object.
     *
     * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
     *   The config factory.
     * @param \Drupal\address_tr\Service\SubdivisionLoaderService $subdivision_loader
     *   The subdivision loader service.
     * @param \Psr\Log\LoggerInterface $logger
     *   The logger.
     */
    public function __construct(
        ConfigFactoryInterface $config_factory,
        SubdivisionLoaderService $subdivision_loader,
        LoggerInterface $logger
    ) {
        $this->configFactory = $config_factory;
        $this->subdivisionLoader = $subdivision_loader;
        $this->logger = $logger;
    }

    /**
     * {@inheritdoc}
     */
    public function getRateForAddress(string $province_code, string $district_code, string $neighbourhood_code): ?Price
    {
        return $this->findRate($this->getAllRates(), $province_code, $district_code, $neighbourhood_code);
    }

    /**
     * {@inheritdoc}
     */
    public function findRate(array $rates, string $province_code, string $district_code, string $neighbourhood_code): ?Price
    {
        if (empty($rates)) {
            return NULL;
        }

        // Try to find the most specific match.
        // Priority: neighbourhood > district > city.
        $neighbourhood_match = NULL;
        $district_match = NULL;
        $city_match = NULL;

        foreach ($rates as $rate) {
            $rate_province = $rate['province_code'] ?? '';
            $rate_district = $rate['district_code'] ?? '';
            $rate_neighbourhood = $rate['neighbourhood_code'] ?? '';

            // Skip if province doesn't match.
            if ($rate_province !== $province_code) {
                continue;
            }

            // Neighbourhood-level match (most specific).
            if (!empty($rate_neighbourhood) && $rate_district === $district_code && $rate_neighbourhood === $neighbourhood_code) {
                $neighbourhood_match = $rate;
                break; // Most specific, no need to continue.
            }

            // District-level match.
            if (!empty($rate_district) && empty($rate_neighbourhood) && $rate_district === $district_code) {
                $district_match = $rate;
            }

            // City-level match (least specific).
            if (empty($rate_district) && empty($rate_neighbourhood)) {
                $city_match = $rate;
            }
        }

        // Return the most specific match found.
        $match = $neighbourhood_match ?? $district_match ?? $city_match;

        if ($match) {
            return new Price($match['price'], $match['currency_code']);
        }

        return NULL;
    }

    /**
     * {@inheritdoc}
     */
    public function getAllRates(): array
    {
        $config = $this->configFactory->get('commerce_courier_shipping.table_rates');
        return $config->get('table_rates') ?: [];
    }

    /**
     * {@inheritdoc}
     */
    public function saveRates(array $rates): void
    {
        $config = $this->configFactory->getEditable('commerce_courier_shipping.table_rates');
        $config->set('table_rates', $rates)->save();
    }

    /**
     * {@inheritdoc}
     */
    public function importFromCsv(string $csv_content): array
    {
        $result = [
            'success' => TRUE,
            'imported' => 0,
            'errors' => [],
        ];

        $lines = explode("\n", trim($csv_content));
        if (empty($lines)) {
            $result['success'] = FALSE;
            $result['errors'][] = 'CSV file is empty.';
            return $result;
        }

        // Parse header.
        $header = str_getcsv(array_shift($lines));
        $expected_headers = ['province_code', 'district_code', 'neighbourhood_code', 'price', 'currency'];

        // Normalize headers.
        $header = array_map('strtolower', array_map('trim', $header));

        // Check if headers match.
        $missing_headers = array_diff($expected_headers, $header);
        if (!empty($missing_headers)) {
            $result['success'] = FALSE;
            $result['errors'][] = 'Missing CSV headers: ' . implode(', ', $missing_headers);
            return $result;
        }

        $rates = [];
        $line_number = 1;

        foreach ($lines as $line) {
            $line_number++;

            if (empty(trim($line))) {
                continue;
            }

            $row = str_getcsv($line);
            $data = array_combine($header, $row);

            // Validate required fields.
            if (empty($data['province_code'])) {
                $result['errors'][] = "Line {$line_number}: Province code is required.";
                continue;
            }

            if (empty($data['price']) || !is_numeric($data['price'])) {
                $result['errors'][] = "Line {$line_number}: Valid price is required.";
                continue;
            }

            if (empty($data['currency'])) {
                $result['errors'][] = "Line {$line_number}: Currency is required.";
                continue;
            }

            $rates[] = [
                'province_code' => trim($data['province_code']),
                'district_code' => trim($data['district_code'] ?? ''),
                'neighbourhood_code' => trim($data['neighbourhood_code'] ?? ''),
                'price' => trim($data['price']),
                'currency_code' => strtoupper(trim($data['currency'])),
            ];

            $result['imported']++;
        }

        if (!empty($rates)) {
            $this->saveRates($rates);
        }

        if (!empty($result['errors'])) {
            $result['success'] = FALSE;
        }

        return $result;
    }

    /**
     * {@inheritdoc}
     */
    public function exportToCsv(): string
    {
        $rates = $this->getAllRates();

        $lines = [];
        $lines[] = 'province_code,district_code,neighbourhood_code,price,currency';

        foreach ($rates as $rate) {
            $lines[] = implode(',', [
                $rate['province_code'] ?? '',
                $rate['district_code'] ?? '',
                $rate['neighbourhood_code'] ?? '',
                $rate['price'] ?? '0',
                $rate['currency_code'] ?? 'TRY',
            ]);
        }

        return implode("\n", $lines);
    }

    /**
     * {@inheritdoc}
     */
    public function getProvinceOptions(): array
    {
        $cache_key = 'provinces';
        if (isset($this->subdivisionCache[$cache_key])) {
            return $this->subdivisionCache[$cache_key];
        }

        $data = $this->subdivisionLoader->loadSubdivisions(['TR']);
        $options = ['' => '- Select -'];

        if (!empty($data['subdivisions'])) {
            foreach ($data['subdivisions'] as $code => $subdivision) {
                $options[$code] = $subdivision['name'] ?? $subdivision['local_name'] ?? $code;
            }
        }

        $this->subdivisionCache[$cache_key] = $options;
        return $options;
    }

    /**
     * {@inheritdoc}
     */
    public function getDistrictOptions(string $province_code): array
    {
        if (empty($province_code)) {
            return ['' => '- Select province first -'];
        }

        $cache_key = 'districts:' . $province_code;
        if (isset($this->subdivisionCache[$cache_key])) {
            return $this->subdivisionCache[$cache_key];
        }

        $data = $this->subdivisionLoader->loadSubdivisions(['TR', $province_code]);
        $options = ['' => '- Any district -'];

        if (!empty($data['subdivisions'])) {
            foreach ($data['subdivisions'] as $code => $subdivision) {
                $options[$code] = $subdivision['name'] ?? $subdivision['local_name'] ?? $code;
            }
        }

        $this->subdivisionCache[$cache_key] = $options;
        return $options;
    }

    /**
     * {@inheritdoc}
     */
    public function getNeighbourhoodOptions(string $province_code, string $district_code): array
    {
        if (empty($province_code) || empty($district_code)) {
            return ['' => '- Select district first -'];
        }

        $cache_key = 'neighbourhoods:' . $province_code . ':' . $district_code;
        if (isset($this->subdivisionCache[$cache_key])) {
            return $this->subdivisionCache[$cache_key];
        }

        // Build the parent for neighbourhood lookup.
        $district_id = $province_code . '-' . $district_code;
        $data = $this->subdivisionLoader->loadSubdivisions(['TR', $province_code, $district_id]);
        $options = ['' => '- Any neighbourhood -'];

        if (!empty($data['subdivisions'])) {
            foreach ($data['subdivisions'] as $code => $subdivision) {
                $options[$code] = $subdivision['name'] ?? $code;
            }
        }

        $this->subdivisionCache[$cache_key] = $options;
        return $options;
    }
}
