<?php

namespace Drupal\commerce_courier_shipping\Service;

/**
 * Interface for the price modifier service.
 */
interface PriceModifierServiceInterface
{

    /**
     * Gets the combined multiplier for a given datetime.
     *
     * All applicable multipliers are multiplied together (stacking).
     *
     * @param \DateTimeInterface|null $datetime
     *   The datetime to calculate multiplier for. Defaults to current time.
     *
     * @return float
     *   The combined multiplier (e.g., 1.0 for no modification, 1.5 for 50% extra).
     */
    public function getMultiplier(?\DateTimeInterface $datetime = NULL): float;

    /**
     * Calculates the combined multiplier using provided modifiers.
     *
     * @param \DateTimeInterface $datetime
     *   The datetime to check.
     * @param array $daily_modifiers
     *   Array of daily modifier configs.
     * @param array $date_modifiers
     *   Array of date modifier configs.
     * @param array $hour_modifiers
     *   Array of hour modifier configs.
     *
     * @return float
     *   The calculated multiplier.
     */
    public function calculateMultiplier(\DateTimeInterface $datetime, array $daily_modifiers, array $date_modifiers, array $hour_modifiers): float;

    /**
     * Gets the daily multiplier for a given datetime.
     *
     * @param \DateTimeInterface $datetime
     *   The datetime to check.
     *
     * @return float
     *   The daily multiplier.
     */
    public function getDailyMultiplier(\DateTimeInterface $datetime): float;

    /**
     * Gets the date multiplier for a given datetime.
     *
     * @param \DateTimeInterface $datetime
     *   The datetime to check.
     *
     * @return float
     *   The date multiplier.
     */
    public function getDateMultiplier(\DateTimeInterface $datetime): float;

    /**
     * Gets the hour multiplier for a given datetime.
     *
     * @param \DateTimeInterface $datetime
     *   The datetime to check.
     *
     * @return float
     *   The hour multiplier.
     */
    public function getHourMultiplier(\DateTimeInterface $datetime): float;

    /**
     * Gets all daily multipliers configuration.
     *
     * @return array
     *   Array of daily multiplier configurations.
     */
    public function getDailyMultipliers(): array;

    /**
     * Gets all date multipliers configuration.
     *
     * @return array
     *   Array of date multiplier configurations.
     */
    public function getDateMultipliers(): array;

    /**
     * Gets all hour multipliers configuration.
     *
     * @return array
     *   Array of hour multiplier configurations.
     */
    public function getHourMultipliers(): array;

    /**
     * Saves all price modifier configurations.
     *
     * @param array $daily_multipliers
     *   Daily multipliers configuration.
     * @param array $date_multipliers
     *   Date multipliers configuration.
     * @param array $hour_multipliers
     *   Hour multipliers configuration.
     */
    public function saveModifiers(array $daily_multipliers, array $date_multipliers, array $hour_multipliers): void;
}
