<?php

namespace Drupal\commerce_courier_shipping\Service;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Psr\Log\LoggerInterface;

/**
 * Service for calculating price modifiers based on time.
 */
class PriceModifierService implements PriceModifierServiceInterface
{

    /**
     * The config factory.
     *
     * @var \Drupal\Core\Config\ConfigFactoryInterface
     */
    protected ConfigFactoryInterface $configFactory;

    /**
     * The time service.
     *
     * @var \Drupal\Component\Datetime\TimeInterface
     */
    protected TimeInterface $time;

    /**
     * The logger.
     *
     * @var \Psr\Log\LoggerInterface
     */
    protected LoggerInterface $logger;

    /**
     * Constructs a PriceModifierService object.
     *
     * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
     *   The config factory.
     * @param \Drupal\Component\Datetime\TimeInterface $time
     *   The time service.
     * @param \Psr\Log\LoggerInterface $logger
     *   The logger.
     */
    public function __construct(
        ConfigFactoryInterface $config_factory,
        TimeInterface $time,
        LoggerInterface $logger
    ) {
        $this->configFactory = $config_factory;
        $this->time = $time;
        $this->logger = $logger;
    }

    /**
     * {@inheritdoc}
     */
    public function getMultiplier(?\DateTimeInterface $datetime = NULL): float
    {
        if ($datetime === NULL) {
            $datetime = new \DateTime('@' . $this->time->getRequestTime());
            $datetime->setTimezone(new \DateTimeZone(date_default_timezone_get()));
        }

        return $this->calculateMultiplier(
            $datetime,
            $this->getDailyMultipliers(),
            $this->getDateMultipliers(),
            $this->getHourMultipliers()
        );
    }

    /**
     * {@inheritdoc}
     */
    public function calculateMultiplier(\DateTimeInterface $datetime, array $daily_modifiers, array $date_modifiers, array $hour_modifiers): float
    {
        // Get all applicable multipliers and multiply them together.
        // We pass the specific arrays to the helper methods.
        // Note: getDailyMultiplier et al currently read from global config if called directly.
        // To reusing the logic we need those methods to accept an optional array argument,
        // OR we duplicate the logic here, OR we extract internal helpers.
        // Let's extract internal static logic or modify the protected methods to take arguments.
        // Modifying the public methods signature would be a breaking change if they are on interface.
        // The interface defines getDailyMultiplier(\DateTimeInterface $datetime): float
        // So we can't easily change those without changing interface.
        // Buuut, we are in charge of the interface.
        // Let's create protected helpers that take the array.

        $daily_multiplier = $this->calculateDailyMultiplier($datetime, $daily_modifiers);
        $date_multiplier = $this->calculateDateMultiplier($datetime, $date_modifiers);
        $hour_multiplier = $this->calculateHourMultiplier($datetime, $hour_modifiers);

        $combined = $daily_multiplier * $date_multiplier * $hour_multiplier;

        $this->logger->debug('Price multiplier for @datetime: daily=@daily, date=@date, hour=@hour, combined=@combined', [
            '@datetime' => $datetime->format('Y-m-d H:i:s'),
            '@daily' => $daily_multiplier,
            '@date' => $date_multiplier,
            '@hour' => $hour_multiplier,
            '@combined' => $combined,
        ]);

        return $combined;
    }

    /**
     * {@inheritdoc}
     */
    public function getDailyMultiplier(\DateTimeInterface $datetime): float
    {
        return $this->calculateDailyMultiplier($datetime, $this->getDailyMultipliers());
    }

    /**
     * Calculates daily multiplier from provided config.
     */
    protected function calculateDailyMultiplier(\DateTimeInterface $datetime, array $multipliers): float
    {
        if (empty($multipliers)) {
            return 1.0;
        }

        // Get day of week (1 = Monday, 7 = Sunday).
        $day_of_week = (int) $datetime->format('N');
        $combined_multiplier = 1.0;

        foreach ($multipliers as $modifier) {
            $days = $modifier['days'] ?? [];
            $multiplier = (float) ($modifier['multiplier'] ?? 1.0);

            if (in_array($day_of_week, $days, TRUE)) {
                $combined_multiplier *= $multiplier;
            }
        }

        return $combined_multiplier;
    }

    /**
     * {@inheritdoc}
     */
    public function getDateMultiplier(\DateTimeInterface $datetime): float
    {
        return $this->calculateDateMultiplier($datetime, $this->getDateMultipliers());
    }

    /**
     * Calculates date multiplier from provided config.
     */
    protected function calculateDateMultiplier(\DateTimeInterface $datetime, array $multipliers): float
    {
        if (empty($multipliers)) {
            return 1.0;
        }

        $current_date = $datetime->format('Y-m-d');
        $combined_multiplier = 1.0;

        foreach ($multipliers as $modifier) {
            $dates = $modifier['dates'] ?? [];
            $multiplier = (float) ($modifier['multiplier'] ?? 1.0);

            if (in_array($current_date, $dates, TRUE)) {
                $combined_multiplier *= $multiplier;
            }
        }

        return $combined_multiplier;
    }

    /**
     * {@inheritdoc}
     */
    public function getHourMultiplier(\DateTimeInterface $datetime): float
    {
        return $this->calculateHourMultiplier($datetime, $this->getHourMultipliers());
    }

    /**
     * Calculates hour multiplier from provided config.
     */
    protected function calculateHourMultiplier(\DateTimeInterface $datetime, array $multipliers): float
    {
        if (empty($multipliers)) {
            return 1.0;
        }

        $current_time = $datetime->format('H:i');
        $combined_multiplier = 1.0;

        foreach ($multipliers as $modifier) {
            $start_hour = $modifier['start_hour'] ?? '00:00';
            $end_hour = $modifier['end_hour'] ?? '23:59';
            $multiplier = (float) ($modifier['multiplier'] ?? 1.0);

            if ($this->isTimeInRange($current_time, $start_hour, $end_hour)) {
                $combined_multiplier *= $multiplier;
            }
        }

        return $combined_multiplier;
    }

    /**
     * Checks if a time is within a given range.
     *
     * Handles overnight ranges (e.g., 21:00 to 06:00).
     *
     * @param string $time
     *   The time to check (H:i format).
     * @param string $start
     *   The start time (H:i format).
     * @param string $end
     *   The end time (H:i format).
     *
     * @return bool
     *   TRUE if time is within range, FALSE otherwise.
     */
    protected function isTimeInRange(string $time, string $start, string $end): bool
    {
        // Normalize to ensure consistent format.
        $time = substr($time, 0, 5);
        $start = substr($start, 0, 5);
        $end = substr($end, 0, 5);

        // Handle overnight ranges (e.g., 21:00 to 06:00).
        if ($start > $end) {
            // If current time is after start OR before end, it's in range.
            return $time >= $start || $time < $end;
        }

        // Normal range (same day).
        return $time >= $start && $time < $end;
    }

    /**
     * {@inheritdoc}
     */
    public function getDailyMultipliers(): array
    {
        $config = $this->configFactory->get('commerce_courier_shipping.price_modifiers');
        return $config->get('daily_multipliers') ?: [];
    }

    /**
     * {@inheritdoc}
     */
    public function getDateMultipliers(): array
    {
        $config = $this->configFactory->get('commerce_courier_shipping.price_modifiers');
        return $config->get('date_multipliers') ?: [];
    }

    /**
     * {@inheritdoc}
     */
    public function getHourMultipliers(): array
    {
        $config = $this->configFactory->get('commerce_courier_shipping.price_modifiers');
        return $config->get('hour_multipliers') ?: [];
    }

    /**
     * {@inheritdoc}
     */
    public function saveModifiers(array $daily_multipliers, array $date_multipliers, array $hour_multipliers): void
    {
        $config = $this->configFactory->getEditable('commerce_courier_shipping.price_modifiers');
        $config->set('daily_multipliers', $daily_multipliers);
        $config->set('date_multipliers', $date_multipliers);
        $config->set('hour_multipliers', $hour_multipliers);
        $config->save();
    }
}
