<?php

namespace Drupal\commerce_courier_shipping\Service;

use Psr\Log\LoggerInterface;

/**
 * Service for matching addresses against enabled locations.
 */
class LocationMatcherService implements LocationMatcherServiceInterface
{

    /**
     * The logger.
     *
     * @var \Psr\Log\LoggerInterface
     */
    protected LoggerInterface $logger;

    /**
     * Constructs a LocationMatcherService object.
     *
     * @param \Psr\Log\LoggerInterface $logger
     *   The logger.
     */
    public function __construct(LoggerInterface $logger)
    {
        $this->logger = $logger;
    }

    /**
     * {@inheritdoc}
     */
    public function isLocationEnabled(array $enabled_locations, string $province_code, string $district_code, string $neighbourhood_code = ''): bool
    {
        // If no locations configured, allow all locations.
        if (empty($enabled_locations)) {
            return TRUE;
        }

        // Check each location entry.
        foreach ($enabled_locations as $location) {
            if (($location['province'] ?? '') !== $province_code) {
                continue;
            }

            // If all_districts is true, the entire province is enabled.
            if (!empty($location['all_districts'])) {
                return TRUE;
            }

            // Check specific districts.
            $districts = $location['districts'] ?? [];
            if (!in_array($district_code, $districts)) {
                continue;
            }

            // District matches. Now check neighbourhoods.
            $all_district_neighbourhoods = $location['all_district_neighbourhoods'] ?? [];
            $neighbourhoods = $location['neighbourhoods'] ?? [];

            // If no neighbourhoods specified for this district, all neighbourhoods are enabled.
            if (empty($neighbourhoods) && empty($all_district_neighbourhoods)) {
                return TRUE;
            }

            // If "All [District]" is selected for this district, enable.
            if (in_array($district_code, $all_district_neighbourhoods)) {
                return TRUE;
            }

            // Check specific neighbourhoods.
            if (in_array($neighbourhood_code, $neighbourhoods)) {
                return TRUE;
            }
        }

        return FALSE;
    }
}
