<?php

namespace Drupal\commerce_courier_shipping\Service;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use GuzzleHttp\ClientInterface;
use Psr\Log\LoggerInterface;

/**
 * Service for calculating distances using Google Routes API.
 */
class DistanceCalculator implements DistanceCalculatorInterface
{

    /**
     * The HTTP client.
     *
     * @var \GuzzleHttp\ClientInterface
     */
    protected ClientInterface $httpClient;

    /**
     * The config factory.
     *
     * @var \Drupal\Core\Config\ConfigFactoryInterface
     */
    protected ConfigFactoryInterface $configFactory;

    /**
     * The cache backend.
     *
     * @var \Drupal\Core\Cache\CacheBackendInterface
     */
    protected CacheBackendInterface $cache;

    /**
     * The logger.
     *
     * @var \Psr\Log\LoggerInterface
     */
    protected LoggerInterface $logger;

    /**
     * Constructs a DistanceCalculator object.
     *
     * @param \GuzzleHttp\ClientInterface $http_client
     *   The HTTP client.
     * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
     *   The config factory.
     * @param \Drupal\Core\Cache\CacheBackendInterface $cache
     *   The cache backend.
     * @param \Psr\Log\LoggerInterface $logger
     *   The logger.
     */
    public function __construct(
        ClientInterface $http_client,
        ConfigFactoryInterface $config_factory,
        CacheBackendInterface $cache,
        LoggerInterface $logger
    ) {
        $this->httpClient = $http_client;
        $this->configFactory = $config_factory;
        $this->cache = $cache;
        $this->logger = $logger;
    }

    /**
     * {@inheritdoc}
     */
    public function calculateDistance(string $store_address, string $destination_address): ?int
    {
        $config = $this->configFactory->get('commerce_courier_shipping.settings');
        $api_key = $config->get('google_api_key');

        if (empty($api_key)) {
            $this->logger->error('Google API key is not configured for courier shipping.');
            return NULL;
        }

        // Create cache key from addresses.
        $cache_key = 'commerce_courier_shipping:distance:' . md5($store_address . '|' . $destination_address);

        // Check cache first.
        $cached = $this->cache->get($cache_key);
        if ($cached) {
            return $cached->data;
        }

        try {
            $distance = $this->callRoutesApi($store_address, $destination_address, $api_key);

            if ($distance !== NULL) {
                // Cache the result.
                $cache_duration = $config->get('api_cache_duration') ?: 86400;
                $this->cache->set($cache_key, $distance, \Drupal::time()->getRequestTime() + $cache_duration);
            }

            return $distance;
        } catch (\Exception $e) {
            $this->logger->error('Error calculating distance: @message', ['@message' => $e->getMessage()]);
            return NULL;
        }
    }

    /**
     * Calls the Google Routes API to calculate distance.
     *
     * @param string $origin
     *   The origin address.
     * @param string $destination
     *   The destination address.
     * @param string $api_key
     *   The Google API key.
     *
     * @return int|null
     *   The distance in kilometers (ceiled), or NULL on error.
     */
    protected function callRoutesApi(string $origin, string $destination, string $api_key): ?int
    {
        $url = 'https://routes.googleapis.com/directions/v2:computeRoutes';

        $body = [
            'origin' => [
                'address' => $origin,
            ],
            'destination' => [
                'address' => $destination,
            ],
            'travelMode' => 'DRIVE',
            'routingPreference' => 'TRAFFIC_AWARE',
            'languageCode' => 'tr-TR',
            'units' => 'METRIC',
        ];

        $response = $this->httpClient->request('POST', $url, [
            'headers' => [
                'Content-Type' => 'application/json',
                'X-Goog-Api-Key' => $api_key,
                'X-Goog-FieldMask' => 'routes.distanceMeters',
            ],
            'json' => $body,
        ]);

        $data = json_decode($response->getBody()->getContents(), TRUE);

        if (empty($data['routes'][0]['distanceMeters'])) {
            $this->logger->warning('No route found between @origin and @destination', [
                '@origin' => $origin,
                '@destination' => $destination,
            ]);
            return NULL;
        }

        // Convert meters to kilometers and ceil.
        $distance_meters = $data['routes'][0]['distanceMeters'];
        $distance_km = (int) ceil($distance_meters / 1000);

        $this->logger->info('Calculated distance: @distance km from @origin to @destination', [
            '@distance' => $distance_km,
            '@origin' => $origin,
            '@destination' => $destination,
        ]);

        return $distance_km;
    }

    /**
     * {@inheritdoc}
     */
    public function formatAddress(array $address): string
    {
        $parts = [];

        if (!empty($address['address_line1'])) {
            $parts[] = $address['address_line1'];
        }
        if (!empty($address['address_line2'])) {
            $parts[] = $address['address_line2'];
        }
        if (!empty($address['dependent_locality'])) {
            $parts[] = $address['dependent_locality'];
        }
        if (!empty($address['locality'])) {
            $parts[] = $address['locality'];
        }
        if (!empty($address['administrative_area'])) {
            $parts[] = $address['administrative_area'];
        }
        if (!empty($address['postal_code'])) {
            $parts[] = $address['postal_code'];
        }
        if (!empty($address['country_code'])) {
            $parts[] = $address['country_code'];
        }

        return implode(', ', $parts);
    }
}
