<?php

namespace Drupal\commerce_courier_shipping\Plugin\Commerce\ShippingMethod;

use Drupal\address_tr\Service\AddressLabelResolver;
use Drupal\commerce_courier_shipping\Service\DistanceCalculatorInterface;
use Drupal\commerce_courier_shipping\Service\TableRatesServiceInterface;
use Drupal\commerce_courier_shipping\Service\PriceModifierServiceInterface;
use Drupal\commerce_courier_shipping\Service\LocationMatcherServiceInterface;
use Drupal\commerce_price\Price;
use Drupal\commerce_shipping\Entity\ShipmentInterface;
use Drupal\commerce_shipping\PackageTypeManagerInterface;
use Drupal\commerce_shipping\Plugin\Commerce\ShippingMethod\ShippingMethodBase;
use Drupal\commerce_shipping\ShippingRate;
use Drupal\commerce_shipping\ShippingService;
use Drupal\Core\Form\FormStateInterface;
use Drupal\state_machine\WorkflowManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides the Courier Delivery shipping method.
 *
 * @CommerceShippingMethod(
 *   id = "courier_delivery",
 *   label = @Translation("Courier Delivery"),
 *   services = {
 *     "default" = @Translation("Standard")
 *   },
 * )
 */
class CourierDelivery extends ShippingMethodBase
{

    /**
     * The distance calculator service.
     *
     * @var \Drupal\commerce_courier_shipping\Service\DistanceCalculatorInterface
     */
    protected $distanceCalculator;

    /**
     * The table rates service.
     *
     * @var \Drupal\commerce_courier_shipping\Service\TableRatesServiceInterface
     */
    protected $tableRatesService;

    /**
     * The price modifier service.
     *
     * @var \Drupal\commerce_courier_shipping\Service\PriceModifierServiceInterface
     */
    protected $priceModifierService;

    /**
     * The location matcher service.
     *
     * @var \Drupal\commerce_courier_shipping\Service\LocationMatcherServiceInterface
     */
    protected $locationMatcherService;

    /**
     * The address label resolver.
     *
     * @var \Drupal\address_tr\Service\AddressLabelResolver
     */
    protected $addressLabelResolver;

    /**
     * Constructs a CourierDelivery object.
     */
    public function __construct(
        array $configuration,
        $plugin_id,
        $plugin_definition,
        PackageTypeManagerInterface $package_type_manager,
        WorkflowManagerInterface $workflow_manager,
        DistanceCalculatorInterface $distance_calculator,
        TableRatesServiceInterface $table_rates_service,
        PriceModifierServiceInterface $price_modifier_service,
        LocationMatcherServiceInterface $location_matcher_service,
        AddressLabelResolver $address_label_resolver
    ) {
        parent::__construct($configuration, $plugin_id, $plugin_definition, $package_type_manager, $workflow_manager);

        $this->distanceCalculator = $distance_calculator;
        $this->tableRatesService = $table_rates_service;
        $this->priceModifierService = $price_modifier_service;
        $this->locationMatcherService = $location_matcher_service;
        $this->addressLabelResolver = $address_label_resolver;

        $this->services['default'] = new ShippingService('default', $this->configuration['rate_label']);
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
    {
        return new static(
            $configuration,
            $plugin_id,
            $plugin_definition,
            $container->get('plugin.manager.commerce_package_type'),
            $container->get('plugin.manager.workflow'),
            $container->get('commerce_courier_shipping.distance_calculator'),
            $container->get('commerce_courier_shipping.table_rates'),
            $container->get('commerce_courier_shipping.price_modifier'),
            $container->get('commerce_courier_shipping.location_matcher'),
            $container->get('address_tr.label_resolver')
        );
    }

    /**
     * {@inheritdoc}
     */
    public function defaultConfiguration()
    {
        return [
            'rate_label' => 'Courier Delivery',
            'rate_description' => '',
            'handling_price' => NULL,
            'minimum_distance' => 1,
            'distance_rate' => NULL,
            'free_shipping_threshold' => NULL,
            'free_shipping_percentage' => '',
            'free_shipping_logic' => 'or',
            'services' => ['default'],
            // Per-method enabled locations: {province_code: TRUE} or {province_code: {district_code: TRUE}}
            'enabled_locations' => [],
            // Per-method table rates: array of rate definitions
            'table_rates' => [],
            // Per-method price modifiers
            'price_modifiers' => [
                'daily_multipliers' => [],
                'date_multipliers' => [],
                'hour_multipliers' => [],
            ],
        ] + parent::defaultConfiguration();
    }

    /**
     * {@inheritdoc}
     */
    public function buildConfigurationForm(array $form, FormStateInterface $form_state)
    {
        $form = parent::buildConfigurationForm($form, $form_state);

        $form['rate_label'] = [
            '#type' => 'textfield',
            '#title' => $this->t('Rate label'),
            '#description' => $this->t('Shown to customers when selecting the rate.'),
            '#default_value' => $this->configuration['rate_label'],
            '#required' => TRUE,
        ];

        $form['rate_description'] = [
            '#type' => 'textfield',
            '#title' => $this->t('Rate description'),
            '#description' => $this->t('Provides additional details about the rate to the customer.'),
            '#default_value' => $this->configuration['rate_description'],
        ];

        // Distance-based pricing settings.
        $form['distance'] = [
            '#type' => 'details',
            '#title' => $this->t('Distance-based Pricing'),
            '#open' => FALSE,
            '#weight' => 30,
        ];

        $handling_price = $this->configuration['handling_price'];
        if (isset($handling_price) && !isset($handling_price['number'], $handling_price['currency_code'])) {
            $handling_price = NULL;
        }

        $form['distance']['handling_price'] = [
            '#type' => 'commerce_price',
            '#title' => $this->t('Handling price'),
            '#description' => $this->t('Base price added to all distance-based calculations.'),
            '#default_value' => $handling_price,
            '#required' => TRUE,
        ];

        $form['distance']['minimum_distance'] = [
            '#type' => 'number',
            '#title' => $this->t('Minimum distance (km)'),
            '#description' => $this->t('Minimum distance used for calculation. If actual distance is less, this value is used.'),
            '#default_value' => $this->configuration['minimum_distance'],
            '#min' => 0,
            '#required' => TRUE,
        ];

        $distance_rate = $this->configuration['distance_rate'];
        if (isset($distance_rate) && !isset($distance_rate['number'], $distance_rate['currency_code'])) {
            $distance_rate = NULL;
        }

        $form['distance']['distance_rate'] = [
            '#type' => 'commerce_price',
            '#title' => $this->t('Distance rate (per km)'),
            '#description' => $this->t('Price charged per kilometer.'),
            '#default_value' => $distance_rate,
            '#required' => TRUE,
        ];

        // Free shipping settings.
        $form['free_shipping'] = [
            '#type' => 'details',
            '#title' => $this->t('Free Shipping'),
            '#open' => FALSE,
            '#weight' => 50,
        ];

        $form['free_shipping']['free_shipping_logic'] = [
            '#type' => 'select',
            '#title' => $this->t('Free shipping logic'),
            '#options' => [
                'or' => $this->t('OR - Free if ANY condition is met'),
                'and' => $this->t('AND - Free if ALL conditions are met'),
            ],
            '#default_value' => $this->configuration['free_shipping_logic'],
        ];

        $free_threshold = $this->configuration['free_shipping_threshold'];
        if (isset($free_threshold) && !isset($free_threshold['number'], $free_threshold['currency_code'])) {
            $free_threshold = NULL;
        }

        $form['free_shipping']['free_shipping_threshold'] = [
            '#type' => 'commerce_price',
            '#title' => $this->t('Order total threshold'),
            '#description' => $this->t('Free shipping if order total is greater than or equal to this amount. Leave empty to disable.'),
            '#default_value' => $free_threshold,
            '#required' => FALSE,
        ];

        $form['free_shipping']['free_shipping_percentage'] = [
            '#type' => 'number',
            '#title' => $this->t('Percentage threshold'),
            '#description' => $this->t('Free shipping if courier cost is less than this percentage of order total. Example: 10 means free if shipping < 10% of order total.'),
            '#default_value' => $this->configuration['free_shipping_percentage'],
            '#min' => 0,
            '#max' => 100,
            '#step' => '0.01',
            '#field_suffix' => '%',
        ];

        // Enabled locations settings.
        $form['enabled_locations_wrapper'] = [
            '#type' => 'details',
            '#title' => $this->t('Enabled Locations'),
            '#description' => $this->t('Add locations where this shipping method is available. Leave empty to allow all locations. Select "All Districts" to enable the entire province.'),
            '#open' => FALSE,
            '#weight' => 10,
        ];

        // Get current configuration - stored as array of {province, districts, neighbourhoods}.
        $enabled_locations = $this->configuration['enabled_locations'] ?: [];

        // Province options.
        $province_options = $this->tableRatesService->getProvinceOptions();
        unset($province_options['']);

        // Get number of entries.
        $num_entries = $form_state->get('num_location_entries');
        if ($num_entries === NULL) {
            $num_entries = max(count($enabled_locations), 0);
            $form_state->set('num_location_entries', $num_entries);
        }

        // Table header.
        $form['enabled_locations_wrapper']['locations_table'] = [
            '#type' => 'table',
            '#header' => [
                $this->t('Province'),
                $this->t('Districts'),
                $this->t('Neighbourhoods'),
                $this->t('Operations'),
            ],
            '#empty' => $this->t('No locations configured. Click "Add location" to add one.'),
            '#prefix' => '<div id="locations-table-wrapper">',
            '#suffix' => '</div>',
        ];

        for ($i = 0; $i < $num_entries; $i++) {
            $entry = $enabled_locations[$i] ?? [];
            $saved_province = $entry['province'] ?? '';
            $saved_districts = $entry['districts'] ?? [];
            $saved_neighbourhoods = $entry['neighbourhoods'] ?? [];

            // Get user input during AJAX rebuilds.
            $user_input = $form_state->getUserInput();

            // Check for stored locations (set by add/remove submit handlers).
            $stored_locations = $form_state->get('stored_locations');

            // Initialize current values from stored locations, user input, or saved config.
            $current_province = $saved_province;
            $current_districts = $saved_districts;
            $current_neighbourhoods = $saved_neighbourhoods;

            // Priority 1: Use stored locations (from add/remove handlers).
            if (!empty($stored_locations) && isset($stored_locations[$i])) {
                $stored = $stored_locations[$i];
                if (!empty($stored['province'])) {
                    $current_province = $stored['province'];
                }
                if (!empty($stored['districts'])) {
                    $current_districts = $stored['districts'];
                }
                if (!empty($stored['neighbourhoods'])) {
                    $current_neighbourhoods = $stored['neighbourhoods'];
                }
            }
            // Priority 2: Check user input (for province/district change AJAX).
            elseif (!empty($user_input)) {
                // Use triggering element to find the actual path structure.
                $triggering_element = $form_state->getTriggeringElement();
                $base_path_found = NULL;

                if ($triggering_element) {
                    $parents = $triggering_element['#parents'] ?? [];
                    // Find 'locations_table' in parents to get the base path.
                    $table_key = array_search('locations_table', $parents);
                    if ($table_key !== FALSE) {
                        // Get path up to and including 'locations_table'.
                        $base_path_found = array_slice($parents, 0, $table_key + 1);
                    }
                }

                // Build paths to try - use the found base path first if available.
                $paths_to_try = [];
                if ($base_path_found) {
                    $paths_to_try[] = array_merge($base_path_found, [$i]);
                }
                // Add Commerce shipping method form paths.
                $paths_to_try[] = ['plugin', 0, 'target_plugin_configuration', 'courier_delivery', 'enabled_locations_wrapper', 'locations_table', $i];
                $paths_to_try[] = ['configuration', 'enabled_locations_wrapper', 'locations_table', $i];
                $paths_to_try[] = ['enabled_locations_wrapper', 'locations_table', $i];

                foreach ($paths_to_try as $path) {
                    $row_data = \Drupal\Component\Utility\NestedArray::getValue($user_input, $path);
                    if (!empty($row_data) && is_array($row_data)) {
                        // Found user input for this row.
                        if (isset($row_data['province']) && $row_data['province'] !== '') {
                            $current_province = $row_data['province'];
                        }
                        if (isset($row_data['districts']) && is_array($row_data['districts'])) {
                            $current_districts = $row_data['districts'];
                        }
                        if (isset($row_data['neighbourhoods']) && is_array($row_data['neighbourhoods'])) {
                            $current_neighbourhoods = $row_data['neighbourhoods'];
                        }
                        break;
                    }
                }
            }

            // Province dropdown.
            $form['enabled_locations_wrapper']['locations_table'][$i]['province'] = [
                '#type' => 'select',
                '#title' => $this->t('Province'),
                '#title_display' => 'invisible',
                '#options' => $province_options,
                '#default_value' => $current_province,
                '#empty_option' => $this->t('- Select -'),
                '#ajax' => [
                    'callback' => [get_class($this), 'locationsTableAjaxCallback'],
                    'wrapper' => 'locations-table-wrapper',
                    'event' => 'change',
                ],
            ];

            // Districts - show with "All Districts" option.
            $has_province = !empty($current_province);

            if ($has_province) {
                $district_options = $this->tableRatesService->getDistrictOptions($current_province);
                unset($district_options['']);

                // Add "All Districts" as first option.
                $all_option = ['__all__' => $this->t('All Districts')];
                $district_options = $all_option + $district_options;

                // Check if "All Districts" is selected.
                $all_districts_selected = in_array('__all__', $current_districts);

                $form['enabled_locations_wrapper']['locations_table'][$i]['districts'] = [
                    '#type' => 'select',
                    '#title' => $this->t('Districts'),
                    '#title_display' => 'invisible',
                    '#options' => $district_options,
                    '#default_value' => $current_districts,
                    '#multiple' => TRUE,
                    '#size' => 5,
                    '#attributes' => ['style' => 'min-width: 200px;'],
                    '#ajax' => [
                        'callback' => [get_class($this), 'locationsTableAjaxCallback'],
                        'wrapper' => 'locations-table-wrapper',
                        'event' => 'change',
                    ],
                ];

                // Neighbourhoods - only show if specific districts are selected (not "All Districts").
                $specific_districts = array_filter($current_districts, fn($d) => $d !== '__all__' && !empty($d));

                if (!$all_districts_selected && !empty($specific_districts)) {
                    // Build grouped neighbourhood options.
                    $neighbourhood_options = [];
                    foreach ($specific_districts as $district_code) {
                        $district_label = $district_options[$district_code] ?? $district_code;
                        $district_neighbourhoods = $this->tableRatesService->getNeighbourhoodOptions($current_province, $district_code);
                        unset($district_neighbourhoods['']);

                        if (!empty($district_neighbourhoods)) {
                            // Add "All [District]" at top of group.
                            $group_options = ['__all_' . $district_code . '__' => $this->t('All @district', ['@district' => $district_label])];
                            $group_options += $district_neighbourhoods;
                            $neighbourhood_options[(string) $district_label] = $group_options;
                        }
                    }

                    if (!empty($neighbourhood_options)) {
                        $form['enabled_locations_wrapper']['locations_table'][$i]['neighbourhoods'] = [
                            '#type' => 'select',
                            '#title' => $this->t('Neighbourhoods'),
                            '#title_display' => 'invisible',
                            '#options' => $neighbourhood_options,
                            '#default_value' => $current_neighbourhoods,
                            '#multiple' => TRUE,
                            '#size' => 5,
                            '#attributes' => ['style' => 'min-width: 250px;'],
                        ];
                    } else {
                        $form['enabled_locations_wrapper']['locations_table'][$i]['neighbourhoods'] = [
                            '#markup' => '<em>' . $this->t('No neighbourhoods') . '</em>',
                        ];
                    }
                } else {
                    $form['enabled_locations_wrapper']['locations_table'][$i]['neighbourhoods'] = [
                        '#markup' => '<em>' . $this->t('All districts selected') . '</em>',
                    ];
                }
            } else {
                $form['enabled_locations_wrapper']['locations_table'][$i]['districts'] = [
                    '#markup' => '<em>' . $this->t('Select a province first') . '</em>',
                ];
                $form['enabled_locations_wrapper']['locations_table'][$i]['neighbourhoods'] = [
                    '#markup' => '',
                ];
            }

            // Remove button.
            $form['enabled_locations_wrapper']['locations_table'][$i]['remove'] = [
                '#type' => 'submit',
                '#value' => $this->t('Remove'),
                '#name' => 'remove_location_' . $i,
                '#submit' => [[get_class($this), 'removeLocationEntrySubmit']],
                '#ajax' => [
                    'callback' => [get_class($this), 'locationsTableAjaxCallback'],
                    'wrapper' => 'locations-table-wrapper',
                ],
                '#limit_validation_errors' => [],
                '#attributes' => ['class' => ['button--danger']],
            ];
        }

        // Add location button.
        $form['enabled_locations_wrapper']['add_location'] = [
            '#type' => 'submit',
            '#value' => $this->t('Add location'),
            '#submit' => [[get_class($this), 'addLocationEntrySubmit']],
            '#ajax' => [
                'callback' => [get_class($this), 'locationsTableAjaxCallback'],
                'wrapper' => 'locations-table-wrapper',
            ],
            '#limit_validation_errors' => [],
        ];

        // Table rates settings.
        $form['table_rates_wrapper'] = [
            '#type' => 'details',
            '#title' => $this->t('Table Rates'),
            '#description' => $this->t('Define fixed rates for specific locations. These override distance-based pricing.'),
            '#open' => FALSE,
            '#weight' => 40,
        ];

        $table_rates = $this->configuration['table_rates'] ?: [];
        $num_rates = $form_state->get('num_table_rates');
        if ($num_rates === NULL) {
            $num_rates = max(count($table_rates), 1);
            $form_state->set('num_table_rates', $num_rates);
        }

        $form['table_rates_wrapper']['rates'] = [
            '#type' => 'container',
            '#prefix' => '<div id="table-rates-wrapper">',
            '#suffix' => '</div>',
        ];

        for ($i = 0; $i < $num_rates; $i++) {
            $rate = $table_rates[$i] ?? [];

            $form['table_rates_wrapper']['rates'][$i] = [
                '#type' => 'fieldset',
                '#title' => $this->t('Rate @num', ['@num' => $i + 1]),
            ];

            $form['table_rates_wrapper']['rates'][$i]['province_code'] = [
                '#type' => 'select',
                '#title' => $this->t('Province'),
                '#options' => ['' => $this->t('- Select -')] + $province_options,
                '#default_value' => $rate['province_code'] ?? '',
            ];

            $form['table_rates_wrapper']['rates'][$i]['district_code'] = [
                '#type' => 'textfield',
                '#title' => $this->t('District code'),
                '#description' => $this->t('Leave empty for province-wide rate.'),
                '#default_value' => $rate['district_code'] ?? '',
                '#size' => 20,
            ];

            $form['table_rates_wrapper']['rates'][$i]['neighbourhood_code'] = [
                '#type' => 'textfield',
                '#title' => $this->t('Neighbourhood code'),
                '#description' => $this->t('Leave empty for district-wide rate.'),
                '#default_value' => $rate['neighbourhood_code'] ?? '',
                '#size' => 20,
            ];

            $rate_price = isset($rate['price'], $rate['currency_code'])
                ? ['number' => $rate['price'], 'currency_code' => $rate['currency_code']]
                : NULL;

            $form['table_rates_wrapper']['rates'][$i]['rate_price'] = [
                '#type' => 'commerce_price',
                '#title' => $this->t('Rate'),
                '#default_value' => $rate_price,
                '#required' => FALSE,
            ];
        }

        $form['table_rates_wrapper']['add_rate'] = [
            '#type' => 'submit',
            '#value' => $this->t('Add another rate'),
            '#submit' => [[get_class($this), 'addRateSubmit']],
            '#ajax' => [
                'callback' => [get_class($this), 'tableRatesAjaxCallback'],
                'wrapper' => 'table-rates-wrapper',
            ],
            '#limit_validation_errors' => [],
        ];

        // Price modifiers settings.
        $form['price_modifiers_wrapper'] = [
            '#type' => 'details',
            '#title' => $this->t('Price Modifiers'),
            '#description' => $this->t('Define time-based multipliers for dynamic pricing.'),
            '#open' => FALSE,
            '#weight' => 20,
        ];

        $price_modifiers = $this->configuration['price_modifiers'] ?: [];

        // Daily multipliers (by day of week).
        $form['price_modifiers_wrapper']['daily'] = [
            '#type' => 'details',
            '#title' => $this->t('Daily Multipliers'),
            '#description' => $this->t('Apply multipliers based on day of week.'),
            '#open' => FALSE,
        ];

        $daily_multipliers = $price_modifiers['daily_multipliers'] ?? [];
        $num_daily = $form_state->get('num_daily_modifiers');
        if ($num_daily === NULL) {
            $num_daily = max(count($daily_multipliers), 1);
            $form_state->set('num_daily_modifiers', $num_daily);
        }

        $day_options = [
            1 => $this->t('Monday'),
            2 => $this->t('Tuesday'),
            3 => $this->t('Wednesday'),
            4 => $this->t('Thursday'),
            5 => $this->t('Friday'),
            6 => $this->t('Saturday'),
            7 => $this->t('Sunday'),
        ];

        $form['price_modifiers_wrapper']['daily']['items'] = [
            '#type' => 'container',
            '#prefix' => '<div id="daily-modifiers-wrapper">',
            '#suffix' => '</div>',
        ];

        for ($i = 0; $i < $num_daily; $i++) {
            $modifier = $daily_multipliers[$i] ?? [];

            $form['price_modifiers_wrapper']['daily']['items'][$i] = [
                '#type' => 'container',
                '#attributes' => ['class' => ['container-inline']],
            ];

            $form['price_modifiers_wrapper']['daily']['items'][$i]['days'] = [
                '#type' => 'checkboxes',
                '#title' => $this->t('Days'),
                '#options' => $day_options,
                '#default_value' => $modifier['days'] ?? [],
            ];

            $form['price_modifiers_wrapper']['daily']['items'][$i]['multiplier'] = [
                '#type' => 'number',
                '#title' => $this->t('Multiplier'),
                '#default_value' => $modifier['multiplier'] ?? 1.0,
                '#step' => '0.01',
                '#min' => 0,
            ];
        }

        // Hour multipliers.
        $form['price_modifiers_wrapper']['hourly'] = [
            '#type' => 'details',
            '#title' => $this->t('Hourly Multipliers'),
            '#description' => $this->t('Apply multipliers based on time of day.'),
            '#open' => FALSE,
        ];

        $hour_multipliers = $price_modifiers['hour_multipliers'] ?? [];
        $num_hourly = $form_state->get('num_hourly_modifiers');
        if ($num_hourly === NULL) {
            $num_hourly = max(count($hour_multipliers), 1);
            $form_state->set('num_hourly_modifiers', $num_hourly);
        }

        $form['price_modifiers_wrapper']['hourly']['items'] = [
            '#type' => 'container',
            '#prefix' => '<div id="hourly-modifiers-wrapper">',
            '#suffix' => '</div>',
        ];

        for ($i = 0; $i < $num_hourly; $i++) {
            $modifier = $hour_multipliers[$i] ?? [];

            $form['price_modifiers_wrapper']['hourly']['items'][$i] = [
                '#type' => 'container',
                '#attributes' => ['class' => ['container-inline']],
            ];

            $form['price_modifiers_wrapper']['hourly']['items'][$i]['start_hour'] = [
                '#type' => 'textfield',
                '#title' => $this->t('Start time'),
                '#default_value' => $modifier['start_hour'] ?? '00:00',
                '#size' => 5,
                '#description' => $this->t('HH:MM format'),
            ];

            $form['price_modifiers_wrapper']['hourly']['items'][$i]['end_hour'] = [
                '#type' => 'textfield',
                '#title' => $this->t('End time'),
                '#default_value' => $modifier['end_hour'] ?? '23:59',
                '#size' => 5,
            ];

            $form['price_modifiers_wrapper']['hourly']['items'][$i]['multiplier'] = [
                '#type' => 'number',
                '#title' => $this->t('Multiplier'),
                '#default_value' => $modifier['multiplier'] ?? 1.0,
                '#step' => '0.01',
                '#min' => 0,
            ];
        }

        return $form;
    }

    /**
     * Ajax callback for table rates.
     */
    public static function tableRatesAjaxCallback(array &$form, FormStateInterface $form_state)
    {
        return $form['configuration']['table_rates_wrapper']['rates'];
    }

    /**
     * Submit handler for adding a rate.
     */
    public static function addRateSubmit(array &$form, FormStateInterface $form_state)
    {
        $num_rates = $form_state->get('num_table_rates');
        $form_state->set('num_table_rates', $num_rates + 1);
        $form_state->setRebuild();
    }

    /**
     * Ajax callback for locations table.
     */
    public static function locationsTableAjaxCallback(array &$form, FormStateInterface $form_state)
    {
        // Find the locations table wrapper in the form hierarchy.
        $triggering_element = $form_state->getTriggeringElement();
        $parents = $triggering_element['#array_parents'];

        // Remove elements until we find 'enabled_locations_wrapper'.
        while (!empty($parents) && end($parents) !== 'enabled_locations_wrapper') {
            array_pop($parents);
        }

        if (empty($parents)) {
            // Fallback to common paths.
            return $form['configuration']['enabled_locations_wrapper']['locations_table']
                ?? $form['enabled_locations_wrapper']['locations_table']
                ?? [];
        }

        $parents[] = 'locations_table';
        return \Drupal\Component\Utility\NestedArray::getValue($form, $parents) ?? [];
    }

    /**
     * Helper to get location values from user input.
     */
    protected static function extractLocationValues(array $user_input, FormStateInterface $form_state): array
    {
        $locations = [];

        // Use triggering element's #parents to find the correct path structure.
        // #parents matches the user_input structure (not #array_parents which is form structure).
        $triggering_element = $form_state->getTriggeringElement();
        $base_path = NULL;

        if ($triggering_element) {
            $parents = $triggering_element['#parents'] ?? [];
            // Find 'enabled_locations_wrapper' in parents to get the base path.
            $wrapper_key = array_search('enabled_locations_wrapper', $parents);
            if ($wrapper_key !== FALSE) {
                // Get path up to and including 'enabled_locations_wrapper', then add 'locations_table'.
                $base_path = array_slice($parents, 0, $wrapper_key + 1);
                $base_path[] = 'locations_table';
            }
        }

        // Build paths to try.
        $paths_to_try = [];
        if ($base_path) {
            $paths_to_try[] = $base_path;
        }
        // Add Commerce shipping method form paths (based on network debugging).
        // Path structure: plugin[0][target_plugin_configuration][courier_delivery][...]
        $paths_to_try[] = ['plugin', 0, 'target_plugin_configuration', 'courier_delivery', 'enabled_locations_wrapper', 'locations_table'];
        $paths_to_try[] = ['configuration', 'enabled_locations_wrapper', 'locations_table'];
        $paths_to_try[] = ['enabled_locations_wrapper', 'locations_table'];

        foreach ($paths_to_try as $path) {
            $table_data = \Drupal\Component\Utility\NestedArray::getValue($user_input, $path);
            if (!empty($table_data) && is_array($table_data)) {
                foreach ($table_data as $i => $row) {
                    if (is_array($row)) {
                        $locations[$i] = [
                            'province' => $row['province'] ?? '',
                            'districts' => $row['districts'] ?? [],
                            'neighbourhoods' => $row['neighbourhoods'] ?? [],
                        ];
                    }
                }
                break;
            }
        }

        return $locations;
    }

    /**
     * Submit handler for adding a location entry.
     */
    public static function addLocationEntrySubmit(array &$form, FormStateInterface $form_state)
    {
        // Store current location values before adding new row.
        $user_input = $form_state->getUserInput();
        $current_locations = static::extractLocationValues($user_input, $form_state);
        $form_state->set('stored_locations', $current_locations);

        $num = $form_state->get('num_location_entries');
        $form_state->set('num_location_entries', $num + 1);
        $form_state->setRebuild();
    }

    /**
     * Submit handler for removing a location entry.
     */
    public static function removeLocationEntrySubmit(array &$form, FormStateInterface $form_state)
    {
        // Store current location values before removing.
        $user_input = $form_state->getUserInput();
        $current_locations = static::extractLocationValues($user_input, $form_state);

        $triggering_element = $form_state->getTriggeringElement();
        $button_name = $triggering_element['#name'] ?? '';

        // Extract index from button name (remove_location_X).
        if (preg_match('/remove_location_(\d+)/', $button_name, $matches)) {
            $remove_index = (int) $matches[1];

            // Remove the entry from stored locations.
            unset($current_locations[$remove_index]);
            $current_locations = array_values($current_locations);

            // Decrease count.
            $num = $form_state->get('num_location_entries');
            $form_state->set('num_location_entries', max(0, $num - 1));
        }

        $form_state->set('stored_locations', $current_locations);
        $form_state->setRebuild();
    }

    /**
     * {@inheritdoc}
     */
    public function submitConfigurationForm(array &$form, FormStateInterface $form_state)
    {
        parent::submitConfigurationForm($form, $form_state);

        if (!$form_state->getErrors()) {
            $values = $form_state->getValue($form['#parents']);
            $this->configuration['rate_label'] = $values['rate_label'];
            $this->configuration['rate_description'] = $values['rate_description'];
            $this->configuration['handling_price'] = $values['distance']['handling_price'];
            $this->configuration['minimum_distance'] = (int) $values['distance']['minimum_distance'];
            $this->configuration['distance_rate'] = $values['distance']['distance_rate'];
            $this->configuration['free_shipping_threshold'] = $values['free_shipping']['free_shipping_threshold'];
            $this->configuration['free_shipping_percentage'] = $values['free_shipping']['free_shipping_percentage'];
            $this->configuration['free_shipping_logic'] = $values['free_shipping']['free_shipping_logic'];

            // Process enabled locations from table.
            $enabled_locations = [];
            $locations_table = $values['enabled_locations_wrapper']['locations_table'] ?? [];
            foreach ($locations_table as $entry) {
                $province = $entry['province'] ?? '';
                if (empty($province)) {
                    continue;
                }
                $districts_raw = array_values(array_filter($entry['districts'] ?? []));

                // Check if "All Districts" is selected.
                $all_districts = in_array('__all__', $districts_raw);
                $districts = $all_districts ? [] : array_filter($districts_raw, fn($d) => $d !== '__all__');

                // Process neighbourhoods.
                $neighbourhoods_raw = array_values(array_filter($entry['neighbourhoods'] ?? []));
                $neighbourhoods = [];
                $all_district_neighbourhoods = [];

                foreach ($neighbourhoods_raw as $n) {
                    // Check for "All [District]" patterns like "__all_DISTRICT_CODE__".
                    if (preg_match('/^__all_(.+)__$/', $n, $matches)) {
                        $all_district_neighbourhoods[] = $matches[1];
                    } else {
                        $neighbourhoods[] = $n;
                    }
                }

                $enabled_locations[] = [
                    'province' => $province,
                    'districts' => $districts,
                    'all_districts' => $all_districts,
                    'neighbourhoods' => $neighbourhoods,
                    'all_district_neighbourhoods' => $all_district_neighbourhoods,
                ];
            }
            $this->configuration['enabled_locations'] = $enabled_locations;

            // Process table rates.
            $table_rates = [];
            $rates_input = $values['table_rates_wrapper']['rates'] ?? [];
            foreach ($rates_input as $rate_data) {
                if (empty($rate_data['province_code'])) {
                    continue;
                }
                $rate_price = $rate_data['rate_price'] ?? [];
                if (empty($rate_price['number'])) {
                    continue;
                }
                $table_rates[] = [
                    'province_code' => $rate_data['province_code'],
                    'district_code' => $rate_data['district_code'] ?? '',
                    'neighbourhood_code' => $rate_data['neighbourhood_code'] ?? '',
                    'price' => $rate_price['number'],
                    'currency_code' => $rate_price['currency_code'],
                ];
            }
            $this->configuration['table_rates'] = $table_rates;

            // Process price modifiers.
            $price_modifiers = [
                'daily_multipliers' => [],
                'date_multipliers' => [],
                'hour_multipliers' => [],
            ];

            // Daily multipliers.
            $daily_items = $values['price_modifiers_wrapper']['daily']['items'] ?? [];
            foreach ($daily_items as $item) {
                $days = array_values(array_filter($item['days'] ?? []));
                if (!empty($days) && !empty($item['multiplier'])) {
                    $price_modifiers['daily_multipliers'][] = [
                        'days' => array_map('intval', $days),
                        'multiplier' => (float) $item['multiplier'],
                    ];
                }
            }

            // Hourly multipliers.
            $hourly_items = $values['price_modifiers_wrapper']['hourly']['items'] ?? [];
            foreach ($hourly_items as $item) {
                if (!empty($item['start_hour']) && !empty($item['end_hour']) && !empty($item['multiplier'])) {
                    $price_modifiers['hour_multipliers'][] = [
                        'start_hour' => $item['start_hour'],
                        'end_hour' => $item['end_hour'],
                        'multiplier' => (float) $item['multiplier'],
                    ];
                }
            }

            $this->configuration['price_modifiers'] = $price_modifiers;
        }
    }

    /**
     * {@inheritdoc}
     */
    public function applies(ShipmentInterface $shipment)
    {
        // Check if shipping address is in enabled locations.
        $shipping_profile = $shipment->getShippingProfile();
        if (!$shipping_profile) {
            return FALSE;
        }

        $address = $shipping_profile->get('address')->first();
        if (!$address) {
            return FALSE;
        }

        $address_values = $address->getValue();
        $country_code = $address_values['country_code'] ?? '';

        // Only apply to Turkish addresses.
        if ($country_code !== 'TR') {
            return FALSE;
        }

        $province_code = $address_values['administrative_area'] ?? '';
        $district_code = $address_values['locality'] ?? '';
        $neighbourhood_code = $address_values['dependent_locality'] ?? '';

        return $this->locationMatcherService->isLocationEnabled($this->configuration['enabled_locations'] ?? [], $province_code, $district_code, $neighbourhood_code);
    }

    /**
     * {@inheritdoc}
     */
    public function calculateRates(ShipmentInterface $shipment)
    {
        $rates = [];

        $shipping_profile = $shipment->getShippingProfile();
        if (!$shipping_profile) {
            return $rates;
        }

        $address = $shipping_profile->get('address')->first();
        if (!$address) {
            return $rates;
        }

        $address_values = $address->getValue();
        $province_code = $address_values['administrative_area'] ?? '';
        $district_code = $address_values['locality'] ?? '';
        $neighbourhood_code = $address_values['dependent_locality'] ?? '';

        // Try to get table rate first (most specific wins).
        $rate_amount = $this->tableRatesService->findRate($this->configuration['table_rates'] ?? [], $province_code, $district_code, $neighbourhood_code);
        $rate_source = 'table';

        // If no table rate, calculate distance-based price.
        if ($rate_amount === NULL) {
            $rate_amount = $this->calculateDistancePrice($shipment, $address_values);
            $rate_source = 'distance';
        }

        if ($rate_amount === NULL) {
            // No rate could be calculated.
            return $rates;
        }

        // Apply price modifiers (multipliers) from plugin configuration.
        // Apply price modifiers (multipliers) from plugin configuration.
        $modifiers_config = $this->configuration['price_modifiers'] ?? [];
        $multiplier = $this->priceModifierService->calculateMultiplier(
            new \DateTime(),
            $modifiers_config['daily_multipliers'] ?? [],
            $modifiers_config['date_multipliers'] ?? [],
            $modifiers_config['hour_multipliers'] ?? []
        );

        if ($multiplier != 1.0) {
            $new_number = bcmul($rate_amount->getNumber(), (string) $multiplier, 6);
            $rate_amount = new Price($new_number, $rate_amount->getCurrencyCode());
        }

        // Check free shipping conditions.
        $order = $shipment->getOrder();
        if ($order && $this->qualifiesForFreeShipping($order, $rate_amount)) {
            $rate_amount = new Price('0', $rate_amount->getCurrencyCode());
        }

        $rates[] = new ShippingRate([
            'shipping_method_id' => $this->parentEntity->id(),
            'service' => $this->services['default'],
            'amount' => $rate_amount,
            'description' => $this->configuration['rate_description'],
        ]);

        return $rates;
    }



    /**
     * Calculates the distance-based price.
     *
     * @param \Drupal\commerce_shipping\Entity\ShipmentInterface $shipment
     *   The shipment.
     * @param array $destination_address
     *   The destination address values.
     *
     * @return \Drupal\commerce_price\Price|null
     *   The calculated price, or NULL on error.
     */
    protected function calculateDistancePrice(ShipmentInterface $shipment, array $destination_address): ?Price
    {
        $order = $shipment->getOrder();
        if (!$order) {
            return NULL;
        }

        $store = $order->getStore();
        if (!$store) {
            return NULL;
        }

        // Get store address.
        $store_address_field = $store->get('address');
        if ($store_address_field->isEmpty()) {
            return NULL;
        }

        $store_address = $store_address_field->first()->getValue();

        // Resolve address labels for better geocoding.
        $store_address_resolved = $this->addressLabelResolver->resolveLabels($store_address);
        $destination_resolved = $this->addressLabelResolver->resolveLabels($destination_address);

        // Format addresses for API.
        $store_formatted = $this->distanceCalculator->formatAddress($store_address_resolved + $store_address);
        $destination_formatted = $this->distanceCalculator->formatAddress($destination_resolved + $destination_address);

        // Calculate distance.
        $distance_km = $this->distanceCalculator->calculateDistance($store_formatted, $destination_formatted);

        if ($distance_km === NULL) {
            // Fall back to table rate if distance calculation fails.
            return NULL;
        }

        // Apply minimum distance.
        $minimum_distance = $this->configuration['minimum_distance'] ?: 1;
        $effective_distance = max($distance_km, $minimum_distance);

        // Calculate price: (distance * rate) + handling.
        $handling_price = $this->configuration['handling_price'];
        $distance_rate = $this->configuration['distance_rate'];

        if (!$handling_price || !$distance_rate) {
            return NULL;
        }

        $handling = new Price($handling_price['number'], $handling_price['currency_code']);
        $rate_per_km = new Price($distance_rate['number'], $distance_rate['currency_code']);

        // distance_cost = effective_distance * rate_per_km.
        $distance_cost_number = bcmul((string) $effective_distance, $rate_per_km->getNumber(), 6);
        $distance_cost = new Price($distance_cost_number, $rate_per_km->getCurrencyCode());

        // total = distance_cost + handling.
        $total = $distance_cost->add($handling);

        return $total;
    }

    /**
     * Checks if the order qualifies for free shipping.
     *
     * @param \Drupal\commerce_order\Entity\OrderInterface $order
     *   The order.
     * @param \Drupal\commerce_price\Price $shipping_cost
     *   The calculated shipping cost.
     *
     * @return bool
     *   TRUE if free shipping applies, FALSE otherwise.
     */
    protected function qualifiesForFreeShipping($order, Price $shipping_cost): bool
    {
        $logic = $this->configuration['free_shipping_logic'] ?: 'or';
        $threshold = $this->configuration['free_shipping_threshold'];
        $percentage = $this->configuration['free_shipping_percentage'];

        $threshold_met = FALSE;
        $percentage_met = FALSE;

        // Get order total.
        $order_total = $order->getTotalPrice();
        if (!$order_total) {
            return FALSE;
        }

        // Check order total threshold.
        if (!empty($threshold['number']) && !empty($threshold['currency_code'])) {
            $threshold_price = new Price($threshold['number'], $threshold['currency_code']);
            // Only compare if same currency.
            if ($order_total->getCurrencyCode() === $threshold_price->getCurrencyCode()) {
                $threshold_met = $order_total->greaterThanOrEqual($threshold_price);
            }
        }

        // Check percentage threshold.
        if (!empty($percentage) && is_numeric($percentage)) {
            // Calculate percentage of order total.
            $max_shipping = bcmul($order_total->getNumber(), bcdiv((string) $percentage, '100', 6), 6);
            $max_shipping_price = new Price($max_shipping, $order_total->getCurrencyCode());

            // Free if shipping cost < max_shipping.
            if ($shipping_cost->getCurrencyCode() === $max_shipping_price->getCurrencyCode()) {
                $percentage_met = $shipping_cost->lessThan($max_shipping_price);
            }
        }

        // Apply logic.
        if ($logic === 'and') {
            // Both conditions must be true (if both are configured).
            $has_threshold = !empty($threshold['number']);
            $has_percentage = !empty($percentage);

            if ($has_threshold && $has_percentage) {
                return $threshold_met && $percentage_met;
            } elseif ($has_threshold) {
                return $threshold_met;
            } elseif ($has_percentage) {
                return $percentage_met;
            }
            return FALSE;
        } else {
            // OR logic - either condition is true.
            return $threshold_met || $percentage_met;
        }
    }
}
