<?php

namespace Drupal\commerce_courier_shipping\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure global Courier Shipping settings.
 */
class CourierSettingsForm extends ConfigFormBase
{

    /**
     * {@inheritdoc}
     */
    protected function getEditableConfigNames()
    {
        return ['commerce_courier_shipping.settings'];
    }

    /**
     * {@inheritdoc}
     */
    public function getFormId()
    {
        return 'commerce_courier_shipping_settings';
    }

    /**
     * {@inheritdoc}
     */
    public function buildForm(array $form, FormStateInterface $form_state)
    {
        $config = $this->config('commerce_courier_shipping.settings');

        $form['google_api'] = [
            '#type' => 'fieldset',
            '#title' => $this->t('Google Routes API'),
        ];

        $form['google_api']['google_api_key'] = [
            '#type' => 'password',
            '#title' => $this->t('Google API Key'),
            '#description' => $this->t('Enter your Google API key for the Routes API. Leave empty to keep current value.'),
            '#attributes' => [
                'placeholder' => $config->get('google_api_key') ? $this->t('API key is set') : $this->t('Enter API key'),
            ],
        ];

        $form['google_api']['api_cache_duration'] = [
            '#type' => 'number',
            '#title' => $this->t('Cache Duration (seconds)'),
            '#description' => $this->t('How long to cache distance calculation results. Default: 86400 (24 hours).'),
            '#default_value' => $config->get('api_cache_duration') ?: 86400,
            '#min' => 0,
            '#required' => TRUE,
        ];

        $form['links'] = [
            '#type' => 'fieldset',
            '#title' => $this->t('Configuration'),
        ];

        $form['links']['table_rates'] = [
            '#type' => 'link',
            '#title' => $this->t('Table Rates'),
            '#url' => \Drupal\Core\Url::fromRoute('commerce_courier_shipping.table_rates'),
            '#prefix' => '<div class="admin-link">',
            '#suffix' => '</div>',
        ];

        $form['links']['enabled_locations'] = [
            '#type' => 'link',
            '#title' => $this->t('Enabled Locations'),
            '#url' => \Drupal\Core\Url::fromRoute('commerce_courier_shipping.enabled_locations'),
            '#prefix' => '<div class="admin-link">',
            '#suffix' => '</div>',
        ];

        $form['links']['price_modifiers'] = [
            '#type' => 'link',
            '#title' => $this->t('Price Modifiers'),
            '#url' => \Drupal\Core\Url::fromRoute('commerce_courier_shipping.price_modifiers'),
            '#prefix' => '<div class="admin-link">',
            '#suffix' => '</div>',
        ];

        return parent::buildForm($form, $form_state);
    }

    /**
     * {@inheritdoc}
     */
    public function submitForm(array &$form, FormStateInterface $form_state)
    {
        $config = $this->configFactory->getEditable('commerce_courier_shipping.settings');

        // Only update API key if a new value was provided.
        $new_api_key = $form_state->getValue('google_api_key');
        if (!empty($new_api_key)) {
            $config->set('google_api_key', $new_api_key);
        }

        $config->set('api_cache_duration', $form_state->getValue('api_cache_duration'));
        $config->save();

        parent::submitForm($form, $form_state);
    }
}
